<?php

/**
 * Plexible Theme
 *
 * @package  PlexibleTheme
 */

// Require Composer
require_once __DIR__ . '/dist/vendor/autoload.php';

// Initialize Timber
Timber\Timber::init();

/**
 * Constants
 */
defined('THEME_VERSION') or define('THEME_VERSION', '0.9.3');
defined('THEME_ENV') or define('THEME_ENV', 'production');
defined('VITE_DEV_SERVER') or define('VITE_DEV_SERVER', THEME_ENV === 'development');

/**
 * Theme Functions
 *
 * Searches through `functions/` for any .php files to include
 */
function theme_functions() {
  $dir = get_theme_file_path('functions');

  if (!is_dir($dir)) {
    return;
  }

  $files = glob($dir . '/*.php');
  if (!$files) {
    return;
  }

  foreach ($files as $absolute_path) {
    $relative_path = 'functions/' . basename($absolute_path);
    $filepath = locate_template($relative_path);
    if (!$filepath) {
      trigger_error(sprintf(__('Error locating %s for inclusion'), $relative_path), E_USER_WARNING);
    }
    include_once $filepath;
  }
}
add_action('after_setup_theme', 'theme_functions', 0);

/**
 * Data Helpers
 */
require_once get_template_directory() . '/data/site-info.php';
require_once get_template_directory() . '/data/icons.php';
require_once get_template_directory() . '/data/app.php';

/**
 * Allow SVG
 */
add_filter(
  'wp_check_filetype_and_ext',
  function ($data, $file, $filename, $mimes) {
    global $wp_version;
    if ($wp_version !== '4.7.1') {
      return $data;
    }

    $filetype = wp_check_filetype($filename, $mimes);

    return [
      'ext' => $filetype['ext'],
      'type' => $filetype['type'],
      'proper_filename' => $data['proper_filename'],
    ];
  },
  10,
  4,
);

add_filter('upload_mimes', function ($mimes) {
  $mimes['svg'] = 'image/svg+xml';
  return $mimes;
});

// preview SVG in WordPress Media Uploader (levon)
function fix_svg_mime_type($data, $file, $filename, $mimes, $real_mime = '') {
  if (version_compare($GLOBALS['wp_version'], '5.1.0', '>=')) {
    $dosvg = in_array($real_mime, ['image/svg', 'image/svg+xml']);
  } else {
    $dosvg = '.svg' === strtolower(substr($filename, -4));
  }
  if ($dosvg) {
    if (current_user_can('manage_options')) {
      $data['ext'] = 'svg';
      $data['type'] = 'image/svg+xml';
    } else {
      $data['ext'] = $type_and_ext['type'] = false;
    }
  }
  return $data;
}
add_filter('wp_check_filetype_and_ext', 'fix_svg_mime_type', 10, 5);

/**
 * ACF Schemas
 *
 * NOTE: The order in which these actions are fired is extremely important
 */
require_once get_stylesheet_directory() . '/lib/acf/options.php';
require_once get_stylesheet_directory() . '/lib/acf/schemas/schema-loader.php';

// preload common code + options pages
add_action('acf/init', 'preload_schemas', 5);

// register dynamic cpt's
add_action(
  'acf/init',
  function () {
    register_post_type_for_all_languages();
    register_optional_post_types();
  },
  10,
);

// load all field groups
add_action('acf/init', 'load_schemas', 15);

/**
 * Create Colour Theme
 */
function create_color_theme() {
  // look for cached version first
  $cached_css = get_transient('theme_color_css');

  // if this is cached version and user is not in the admin panel use it
  // if ($cached_css !== false && !is_admin()) {
  if ($cached_css !== false) {
    echo $cached_css;
    return;
  }

  function get_theme_palette(string $theme): array {
    // TODO: move accepted themes to a global control
    if (!in_array($theme, ['light', 'dark'])) {
      return [];
    }

    $all_themes = get_theme_option('theme') ?? [];

    // check if the specific theme exists in $all_themes
    if (empty($all_themes) || !isset($all_themes[$theme])) {
      return [];
    }

    $theme_colors = $all_themes[$theme];
    $color_keys = [
      'primary',
      'secondary',
      'accent',
      'text',
      'text_muted',
      'background',
      'border',
      'caption',
      'link',
      'grey_1',
      'grey_2',
      'grey_3',
      'grey_4',
      'grey_5',
      'grey_6',
      'grey_7',
      'grey_8',
      'grey_9',
      'white',
      'black',
    ];

    $palette = [];
    foreach ($color_keys as $key) {
      if (isset($theme_colors[$key])) {
        $palette["palette-{$theme}-" . str_replace('_', '-', $key)] = $theme_colors[$key];
      }
    }

    return $palette;
  }

  $theme_light_palette = get_theme_palette('light');
  $theme_dark_palette = get_theme_palette('dark');

  function get_acf_font_group($group_name) {
    $typography = get_theme_option('typography');
    $fonts = $typography['fonts'];

    if ($fonts['control']['enable'] && isset($fonts['custom'][$group_name])) {
      $group = $fonts['custom'][$group_name];
      $name = $group['name'];

      $css_var_name = 'font-' . str_replace('_', '-', $group_name);

      return [$css_var_name => $name];
    }

    return [];
  }

  $font_primary = get_acf_font_group('primary');
  $font_secondary = get_acf_font_group('secondary');

  function get_theme_multiplier(string $multiplier_group): array {
    $all_multipliers = get_theme_option($multiplier_group) ?? [];

    if (empty($all_multipliers)) {
      return [];
    }

    $multiplier_keys = ['text', 'space', 'radius'];

    $multipliers = [];

    foreach ($multiplier_keys as $key) {
      if (isset($all_multipliers[$key])) {
        $multipliers['multiplier-' . str_replace('_', '-', $key)] = $all_multipliers[$key];
      }
    }

    return $multipliers;
  }

  $appearance_multipliers = get_theme_multiplier('appearance_multipliers');

  function get_theme_text_weight(string $group): array {
    $typography = get_theme_option('typography');
    $all_weights = $typography[$group] ?? [];

    if (empty($all_weights)) {
      return [];
    }

    $weight_keys = ['light', 'normal', 'medium', 'semibold', 'bold'];

    $weights = [];

    foreach ($weight_keys as $key) {
      if (isset($all_weights[$key])) {
        $weights['text-weight-' . str_replace('_', '-', $key)] = $all_weights[$key];
      }
    }

    return $weights;
  }

  $text_weights = get_theme_text_weight('font_weights');

  $all_generated_variables = array_merge(
    $theme_light_palette,
    $theme_dark_palette,
    $font_primary,
    $font_secondary,
    $text_weights,
    $appearance_multipliers,
  );

  // prepare CSS variables
  $css_variables = [];
  foreach ($all_generated_variables as $var_name => $value) {
    // check for `font-` and surround in quotes to support spaced font names
    if (strpos($var_name, 'font-') !== false) {
      $css_variables[$var_name] = '"' . $value . '"';
      continue;
    }

    if ($value === null || $value === '') {
      continue;
    }

    $css_variables[$var_name] = $value;

    if (is_valid_hex_color($value)) {
      $css_variables[$var_name . '-alpha'] = hexToRgb($value);
    }
  }

  // Generate CSS
  ob_start();
  ?>

    <style type="text/css">
        :host, :root, .e_root {
            <?php foreach ($css_variables as $var_name => $value): ?>
                --<?= esc_attr($var_name) ?>: <?= $value ?>;
            <?php endforeach; ?>
        }
    </style>
    <?php
    $css = ob_get_clean();

    // Cache the CSS for 1 week
    set_transient('theme_color_css', $css, WEEK_IN_SECONDS);

    // Output the CSS
    echo $css;
}

add_action('wp_head', 'create_color_theme'); // font-end
add_action('admin_head', 'create_color_theme'); // editor

function clear_theme_color_cache($post_id) {
  if ($post_id == 'options') {
    delete_transient('theme_color_css');
  }
}

add_action('acf/save_post', 'clear_theme_color_cache');

/**
 * Fonts
 *
 * Enqueue and preload Google Fonts if present
 */
function theme_fonts(): void {
  $typography = get_theme_option('typography') ?? [];
  $fonts = $typography['fonts'] ?? [];

  // bail early if `custom fonts` disabled or no font URLs
  if (
    empty($fonts['control']['enable']) ||
    (empty($fonts['custom']['primary']['link']) && empty($fonts['custom']['secondary']['link']))
  ) {
    return;
  }

  $primary = (string) ($fonts['custom']['primary']['link'] ?? '');
  $secondary = (string) ($fonts['custom']['secondary']['link'] ?? '');

  // helper to preload + enqueue
  $load = fn(string $handle, string $url): array => [
    sprintf('<link rel="preload" href="%s" as="style" id="%s-preload">', esc_url($url), esc_attr($handle)),
    wp_enqueue_style($handle, esc_url($url), [], null) ?: '',
  ];

  $tags = [];

  if ($primary || $secondary) {
    $tags[] = '<link rel="preconnect" href="https://fonts.googleapis.com">';
    $tags[] = '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>';
  }
  if ($primary !== '') {
    $tags = array_merge($tags, $load('theme-primary-font', $primary));
  }
  if ($secondary !== '' && $secondary !== $primary) {
    $tags = array_merge($tags, $load('theme-secondary-font', $secondary));
  }

  echo implode("\n", array_filter($tags));
}
add_action('wp_enqueue_scripts', 'theme_fonts');

/**
 * Favicon
 *
 * Custom front-end favicon with fallback
 */
function theme_favicon(): void {
  if (!function_exists('get_field')) {
    return;
  }

  $favicon = get_theme_option('brand_favicon') ?? '';
  $default = get_template_directory_uri() . '/assets/images/admin-favicon.svg';
  $icons = [];

  if ($favicon !== '') {
    $icons[] = sprintf('<link rel="icon" href="%s" type="image/x-icon">', esc_url($favicon));
    $icons[] = sprintf('<link rel="shortcut icon" href="%s" type="image/x-icon">', esc_url($favicon));
  } else {
    $icons[] = sprintf('<link rel="shortcut icon" href="%s" type="image/x-icon">', esc_url($default));
  }

  echo implode("\n", $icons);
}
add_action('wp_head', 'theme_favicon', 20);

/**
 * Post Types
 *
 * Create, register and handle optional post types
 */
function create_custom_post_type($args = []) {
  $_defaults = [
    'option_prefix' => 'option',
    'supports' => ['title', 'editor', 'thumbnail', 'custom-fields'],
    'menu_position' => 20,
    'default_icon' => 'dashicons-admin-post',
    'taxonomies' => [],
    'template_id' => null,
    'show_in_menu' => true,
  ];

  // merge passed args with defaults
  $params = array_merge_deep($_defaults, $args);

  // retrieve settings from ACF Options
  if ($params['dynamic']) {
    $title = get_theme_option($params['title_id']);
    $slug = get_theme_option($params['slug_id']);
    $icon = get_theme_option($params['icon_id']);
  } else {
    $title = $params['title'];
    $slug = $params['slug'];
    $icon = $params['icon'];
  }

  // fallback if no slug is set
  $post_type_slug = !empty($slug) ? sanitize_title($slug) : sanitize_title($title);

  register_post_type($post_type_slug, [
    'labels' => [
      'name' => !empty($title) ? $title : 'Custom Post Type',
      'singular_name' => !empty($title) ? $title : 'Custom Post Type',
      'add_new' => 'Add New',
      'add_new_item' => 'Add New ' . (!empty($title) ? $title : 'Item'),
      'edit_item' => 'Edit ' . (!empty($title) ? $title : 'Item'),
      'new_item' => 'New ' . (!empty($title) ? $title : 'Item'),
      'view_item' => 'View ' . (!empty($title) ? $title : 'Item'),
      'search_items' => 'Search ' . (!empty($title) ? $title : 'Items'),
      'not_found' => 'No ' . (!empty($title) ? strtolower($title) : 'items') . ' found',
      'not_found_in_trash' => 'No ' . (!empty($title) ? strtolower($title) : 'items') . ' found in Trash',
    ],
    'public' => true,
    'has_archive' => true,
    'publicly_queryable' => true,
    'query_var' => true,
    'rewrite' => [
      'slug' => $post_type_slug,
    ],
    'capability_type' => 'post',
    'hierarchical' => false,
    'supports' => $params['supports'],
    'menu_position' => $params['menu_position'],
    'menu_icon' => !empty($icon) ? $icon : $params['default_icon'],
    'template_id' => $params['template_id'],
    'show_in_rest' => true,
    'show_in_menu' => $params['show_in_menu'],
  ]);

  // register unique taxonomies for the custom post type if provided
  if (!empty($params['taxonomies'])) {
    foreach ($params['taxonomies'] as $taxonomy) {
      $taxonomy_slug = !empty($taxonomy['slug'])
        ? sanitize_title($taxonomy['slug'])
        : sanitize_title($taxonomy['name']);

      register_taxonomy($taxonomy_slug, $post_type_slug, [
        'labels' => [
          'name' => $taxonomy['name'],
          'singular_name' => $taxonomy['singular_name'] ?? $taxonomy['name'],
          'search_items' => 'Search ' . $taxonomy['name'],
          'all_items' => 'All ' . $taxonomy['name'],
          'parent_item' => 'Parent ' . ($taxonomy['singular_name'] ?? $taxonomy['name']),
          'parent_item_colon' => 'Parent ' . ($taxonomy['singular_name'] ?? $taxonomy['name']) . ':',
          'edit_item' => 'Edit ' . ($taxonomy['singular_name'] ?? $taxonomy['name']),
          'update_item' => 'Update ' . ($taxonomy['singular_name'] ?? $taxonomy['name']),
          'add_new_item' => 'Add New ' . ($taxonomy['singular_name'] ?? $taxonomy['name']),
          'new_item_name' => 'New ' . ($taxonomy['singular_name'] ?? $taxonomy['name']) . ' Name',
          'menu_name' => $taxonomy['name'],
        ],
        'hierarchical' => $taxonomy['hierarchical'] ?? true,
        'show_ui' => true,
        'show_admin_column' => true,
        'query_var' => true,
        'rewrite' => ['slug' => $taxonomy_slug],
      ]);
    }
  }
}

// REGISTER POST TYPES
function register_post_type_for_all_languages() {
  if (!function_exists('acf_add_local_field_group') || !function_exists('acf_add_options_page')) {
    return;
  }

  create_custom_post_type([
    'dynamic' => true,
    'title_id' => 'custom_post_type_1_title',
    'slug_id' => 'custom_post_type_1_slug',
    'icon_id' => 'custom_post_type_1_icon',
    'template_id' => 'cpt-1',
  ]);
  create_custom_post_type([
    'dynamic' => true,
    'title_id' => 'custom_post_type_2_title',
    'slug_id' => 'custom_post_type_2_slug',
    'icon_id' => 'custom_post_type_2_icon',
    'template_id' => 'cpt-2',
    'taxonomies' => [
      [
        'name' => 'Categories',
        'singular_name' => 'category',
        'slug' => 'custom_post_type_2_category',
        'hierarchical' => true,
      ],
    ],
  ]);
  create_custom_post_type([
    'dynamic' => true,
    'title_id' => 'custom_post_type_3_title',
    'slug_id' => 'custom_post_type_3_slug',
    'icon_id' => 'custom_post_type_3_icon',
    'template_id' => 'cpt-3',
  ]);
}
// add_action('acf/init', 'register_post_type_for_all_languages', 20);

// REGISTER MODULE CONTROL
function register_optional_post_types() {
  if (!function_exists('acf_add_local_field_group') || !function_exists('acf_add_options_page')) {
    return;
  }

  $modules = get_theme_option('modules');

  if (empty($modules) || !is_array($modules)) {
    error_log('Modules array is empty or invalid');
    return;
  }

  $enabled_modules = [];

  // check for enabled modules
  foreach (MODULES as $key => $module) {
    $control_key = $module['acf_control'];
    if (!empty($modules[$key][$control_key])) {
      $enabled_modules[$key] = $module;
    }
  }

  // if any modules enabled, create parent item and register post type
  if (!empty($enabled_modules)) {
    add_menu_page('Modules', 'Modules', 'edit_posts', 'modules-menu', function () {}, 'dashicons-image-filter', 21);

    add_action(
      'admin_menu',
      function () {
        remove_submenu_page('modules-menu', 'modules-menu');
      },
      999,
    );

    // register each enabled module
    foreach ($enabled_modules as $key => $module) {
      // check if custom title/string is present in module settings
      $title = !empty($modules[$key]['settings']['title']) ? $modules[$key]['settings']['title'] : $module['title'];
      $slug = !empty($modules[$key]['settings']['slug']) ? $modules[$key]['settings']['slug'] : $module['slug'];

      create_custom_post_type([
        'dynamic' => false,
        'title' => $title,
        'slug' => $slug,
        'template_id' => $module['key'] ?? null,
        'show_in_menu' => 'modules-menu',
        'icon' => 'dashicons-admin-post',
        'show_in_nav_menus' => false,
        'taxonomies' => $module['taxonomies'] ?? null,
      ]);

      // include module-specific ACF fields if they exist
      $acf_file = get_stylesheet_directory() . "/lib/acf/schemas/field-groups/modules/{$module['slug']}.php";
      if (file_exists($acf_file)) {
        include_once $acf_file;
      }
    }
  }
}
// add_action('acf/init', 'register_optional_post_types', 20);

// REMOVE MODULE MENU ITEMS
function remove_post_types_from_nav_menus($post_type) {
  if (!$post_type) {
    return $post_type;
  }

  foreach (MODULES as $key => $module) {
    if ($module['slug'] === $post_type->name && (!isset($module['builder']) || $module['builder'] === false)) {
      return null;
    }
  }

  return $post_type;
}
add_filter('nav_menu_meta_box_object', 'remove_post_types_from_nav_menus');

/**
 * Default Template Assignments
 */

// assing templates to cpt's
add_filter('template_include', function ($template) {
  global $post;

  // Check if we're on a singular post type page
  if (is_singular()) {
    $post_type = get_post_type($post);
  }
  // Or if it's an archive page for the custom post type
  elseif (is_post_type_archive()) {
    $post_type = get_post_type();
  } else {
    return $template; // No need to do anything if it's neither singular nor archive
  }

  $post_type_object = get_post_type_object($post_type);

  if ($post_type_object && isset($post_type_object->template_id)) {
    // Get the template_id from the custom post type registration
    $template_id = $post_type_object->template_id;

    // For singular view, use 'single-{template_id}.php'
    if (is_singular()) {
      $custom_template = get_stylesheet_directory() . "/page-templates/single-{$template_id}.php";
    }
    // For archive view, use 'archive-{template_id}.php'
    elseif (is_post_type_archive()) {
      $custom_template = get_stylesheet_directory() . "/page-templates/archive-{$template_id}.php";
    }

    // Check if the custom template exists
    if (file_exists($custom_template)) {
      return $custom_template;
    }
  }

  return $template;
});

// filter out any templates that match the single-* or archive-* pattern
add_filter(
  'theme_templates',
  function ($post_templates, $wp_theme, $post, $post_type) {
    return array_filter(
      $post_templates,
      function ($template_path) {
        return !preg_match('/(single|archive)-.*\.php$/', $template_path);
      },
      ARRAY_FILTER_USE_KEY,
    );
  },
  10,
  4,
);

// set default template for pages on page creation
function set_default_page_template() {
  global $post;
  if (isset($post) && 'page' === $post->post_type && '' === $post->page_template) {
    $post->page_template = 'page-templates/builder.php';

    // force ACF to refresh fields
    update_post_meta($post->ID, '_wp_page_template', 'page-templates/builder.php');
  }
}
add_action('add_meta_boxes', 'set_default_page_template', 1);

// change default template name `page.php`
add_filter('default_page_template_title', function () {
  return 'Empty';
});

/**
 * USER ROLES
 *
 * Modify role names and remove unwanted roles
 */
add_action('admin_init', function () {
  global $wp_roles;
  if (!isset($wp_roles)) {
    $wp_roles = new WP_Roles();
  }
  $wp_roles->roles['administrator']['name'] = 'Plexible Partner';
  $wp_roles->role_names['administrator'] = 'Plexible Partner';
  $wp_roles->roles['editor']['name'] = 'Plexible Admin';
  $wp_roles->role_names['editor'] = 'Plexible Admin';
});

add_action('admin_init', function () {
  remove_role('author');
  remove_role('contributor');
  remove_role('subscriber');
});

/**
 * Editor capabilities
 */
add_action('admin_init', function () {
  $role = get_role('editor');
  $role->add_cap('edit_theme_options');
});

// remove all appearance menu items except `Menus`
add_action(
  'admin_menu',
  function () {
    if (!current_user_can('editor')) {
      return;
    }
    global $submenu;
    if (isset($submenu['themes.php'])) {
      foreach ($submenu['themes.php'] as $key => $item) {
        // Keep only the Menus page
        if ($item[2] !== 'nav-menus.php') {
          unset($submenu['themes.php'][$key]);
        }
      }
    }
  },
  999,
);

// redirect editors away from other appearance pages
add_action('admin_init', function () {
  if (!current_user_can('editor')) {
    return;
  }
  global $pagenow;
  $allowed_appearance_pages = ['nav-menus.php'];
  if ($pagenow === 'themes.php' && !in_array($_GET['page'] ?? '', $allowed_appearance_pages)) {
    wp_redirect(admin_url('nav-menus.php'));
    exit();
  }
});

/**
 * Google Maps API
 */
add_action('acf/init', function () {
  $keys = get_site_info('keys');
  $gmaps_api_key = $keys['google_maps'];
  acf_update_setting('google_api_key', $gmaps_api_key);
});

add_action('wp_footer', function () {
  if (!is_singular()) {
    return;
  }
  $apiData = apply_filters('acf/fields/google_map/api', []);
  $key = $apiData['key'] ?? acf_get_setting('google_api_key');
  if (!$key) {
    return;
  }
  wp_enqueue_script(
    'google-maps-api',
    esc_url("https://maps.googleapis.com/maps/api/js?key={$key}&v=weekly"),
    [],
    null,
    true,
  );

  add_filter(
    'script_loader_tag',
    function ($tag, $handle) {
      return $tag;
    },
    10,
    2,
  );
});

/**
 * AJAX
 */
require_once get_template_directory() . '/ajax/filter-posts-by-term.php';
require_once get_template_directory() . '/ajax/load-more-posts.php';

if (!function_exists('get_primary_taxonomy_for_post_type')) {
  /**
   * Get the primary taxonomy for a given post type.
   *
   * @param string $post_type
   * @return string|null
   */
  function get_primary_taxonomy_for_post_type($post_type) {
    $taxonomies = get_object_taxonomies($post_type, 'objects');

    foreach ($taxonomies as $taxonomy) {
      // Prefer hierarchical (e.g. categories) and public taxonomies
      if ($taxonomy->hierarchical && !empty($taxonomy->public)) {
        return $taxonomy->name;
      }
    }

    // Fallback: return the first taxonomy name, if available
    $first_tax = reset($taxonomies);
    return $first_tax->name ?? null;
  }
}

/**
 * Yoast
 */
add_filter('wpseo_metabox_prio', function () {
  return 'low';
});

/**
 * Icon Library
 */
function add_plexible_icon_picker_tab(array $tabs): array {
  $tabs['plexible-icons_outline'] = 'PLX Icons (Outline)';
  $tabs['plexible-icons_fill'] = 'PLX Icons (Fill)';
  return $tabs;
}
add_filter('acf/fields/icon_picker/tabs', 'add_plexible_icon_picker_tab');

function add_plexible_outline_icons(array $icons): array {
  $icon_library = get_icon_library();
  $acf_icons = [];

  foreach ($icon_library['outline'] as $icon) {
    $acf_icons[] = [
      'url' => $icon_library['base_uri'] . $icon['key'] . '.svg',
      'key' => $icon['key'],
      'label' => $icon['label'],
    ];
  }
  return $acf_icons;
}
add_filter('acf/fields/icon_picker/plexible-icons_outline/icons', 'add_plexible_outline_icons');

function add_plexible_fill_icons(array $icons): array {
  $icon_library = get_icon_library();
  $acf_icons = [];

  foreach ($icon_library['fill'] as $icon) {
    $acf_icons[] = [
      'url' => $icon_library['base_uri'] . $icon['key'] . '.svg',
      'key' => $icon['key'],
      'label' => $icon['label'],
    ];
  }
  return $acf_icons;
}
add_filter('acf/fields/icon_picker/plexible-icons_fill/icons', 'add_plexible_fill_icons');

function allow_iframe_in_kses_post($tags) {
  $tags['iframe'] = array(
    'src'             => true,
    'width'           => true,
    'height'          => true,
    'frameborder'     => true,
    'allow'           => true,
    'allowfullscreen' => true,
    'loading'         => true,
    'referrerpolicy'  => true,
  );
  return $tags;
}
add_filter('wp_kses_allowed_html', 'allow_iframe_in_kses_post', 10, 1);

// Timber setup
class PlexibleTheme extends Timber\Site {
  /* add timber support. */
  public function __construct() {
    add_action('init', [$this, 'register_taxonomies']);
    add_action('init', [$this, 'register_post_types']);
    add_filter('timber/context', [$this, 'add_to_context']);
    add_filter('timber/twig/filters', [$this, 'add_filters']);
    add_filter('timber/twig/functions', [$this, 'add_functions']);
    add_filter('timber/locations', [$this, 'add_locations']);

    parent::__construct();
  }

  public function add_locations(array $locs): array {
    $namespaces = ['components', 'layouts', 'blocks'];

    foreach ($namespaces as $ns) {
      if (!isset($locs[$ns]) || !is_array($locs[$ns])) {
        $locs[$ns] = [];
      }

      $path = get_template_directory() . '/' . $ns;

      if (is_dir($path) && !in_array($path, $locs[$ns], true)) {
        $locs[$ns][] = $path;
      }
    }

    return $locs;
  }

  /** This is where you add some context
   *
   * @param string $context context['this'] Being the Twig's {{ this }}.
   */
  public function add_to_context($context) {
    $context['site'] = $this;

    $context['THEME_ENV'] = THEME_ENV;
    $context['VITE_DEV_SERVER'] = VITE_DEV_SERVER;

    if (\class_exists('ACF')) {
      $context['options'] = get_fields('options');
    }

    return $context;
  }

  /**
   * This is where you can add your custom functions to Twig.
   *
   * @param string $functions : Array of available functions in Twig.
   */
  public function add_filters($filters) {
    // alias of sanitize
    $filters['slugify'] = $filters['sanitize'];

    return $filters;
  }

  /**
   * This is where you can add your custom functions to Twig.
   *
   * @param string $functions : Array of available functions in Twig.
   */
  public function add_functions($functions) {
    $functions['get_context'] = [
      'callable' => function () {
        return Timber::context();
      },
    ];
    $functions['get_options'] = [
      'callable' => function () {
        return get_fields('options');
      },
    ];
    $functions['get_term_by'] = ['callable' => [Timber::class, 'get_term_by']];
    $functions['get_post_by'] = ['callable' => [Timber::class, 'get_post_by']];

    $functions['build_style_string'] = ['callable' => 'build_style_string', 'is_safe' => ['html']];

    return $functions;
  }
}

new PlexibleTheme();
