<?php

/**
 * Render a component, attempting to resolve common file structures
 *
 * - components/path.php
 * - components/path/path.php (if path is a directory name)
 * - components/path/index.php (if path is a directory name)
 *
 * @param string $path   path relative to the 'components' directory.
 * @param array  $params args to pass to the component.
 */

function component(string $path, array $params = []): void {
  $root_path = get_template_directory();
  $base_path = '/components/classic/';

  // normalize path, remove leading/trailing slashes for consistency
  $n_path = trim($path, '/');

  // possible file locations to try, in order of preference
  $possible_files = [
    $n_path, // for components/path.php OR components/folder/path.php
    $n_path . '/' . basename($n_path), // For components/path/path.php
    $n_path . '/index', // For components/path/index.php
  ];

  $found_path = null;
  $debug_relative_paths = [];

  foreach ($possible_files as $file_slug) {
    $verify_path = $root_path . $base_path . $file_slug . '.php';
    $debug_relative_paths[] = ltrim($base_path . $file_slug . '.php', '/');

    if (file_exists($verify_path)) {
      $found_path = $file_slug;
      break;
    }
  }

  if ($found_path) {
    get_template_part("components/classic/{$found_path}", null, $params);
  } else {
    $error_props = [
      'title' => 'Component Not Found',
      'details' => _component_error_details($path, $n_path, $root_path, $debug_relative_paths),
    ];
    get_template_part('components/error-boundary/index', null, $error_props);
  }
}

function _component_error_details($path, $n_path, $root_path, $debug_relative_paths) {
  ob_start(); ?>
    <p>
      Original requested component path: <code><?= esc_html($path) ?></code>
    </p>
    <p>
      Normalized path used for search: <code><?= esc_html($n_path) ?></code>
    </p>
    <p>
      Could not find a template file. Attempted locations (relative to theme root <code><?= esc_html(
        wp_normalize_path($root_path),
      ) ?>/</code>):
    </p>
    <ul style="margin: 0 0 0 20px; padding: 0; list-style-type: '✗ ';">
      <?php if (!empty($debug_relative_paths) && is_array($debug_relative_paths)): ?>
        <?php foreach ($debug_relative_paths as $attempted_path): ?>
          <li><code><?= esc_html(wp_normalize_path($attempted_path ?? '')) ?></code></li>
        <?php endforeach; ?>
      <?php else: ?>
        <li>No specific attempt paths logged or paths data is invalid.</li>
      <?php endif; ?>
    </ul>
  <?php return ob_get_clean();
}
