<?php

/**
 * Gutenberg + ACF + Twig block setup
 */

use Timber\Timber;

// disable autosave globaly
add_action('admin_init', function () {
  wp_deregister_script('autosave');
});

// disable Gutenberg features
remove_action('enqueue_block_editor_assets', 'wp_enqueue_editor_block_directory_assets');
add_filter('should_load_remote_block_patterns', '__return_false');
add_filter('wp_lazy_loading_enabled', '__return_false');

// create block category
function plexible_block_category($categories, $post) {
  array_unshift($categories, [
    'slug' => 'plexible-blocks',
    'title' => __('Plexible Blocks', 'plexible-blocks'),
    'icon' => 'screenoptions',
  ]);
  return $categories;
}
add_filter('block_categories_all', 'plexible_block_category', 10, 2);

// register all blocks in `/blocks` with valid `block.json`
function plexible_block_register() {
  $dir = get_stylesheet_directory() . '/blocks';

  if (!is_dir($dir)) {
    return;
  }

  foreach (new DirectoryIterator($dir) as $item) {
    if ($item->isDot() || !$item->isDir()) {
      continue;
    }

    $json = $item->getPathname() . '/block.json';
    if (file_exists($json)) {
      register_block_type($item->getPathname());
    }
  }
}
add_action('init', 'plexible_block_register');

// callback function to render appropriate twig template
function acf_twig_block_render($attributes, $content = '', $is_preview = false) {
  // derive slug & fetch block’s own fields
  $slug = str_replace('plexible-blocks/', '', $attributes['name']);
  $block_fields = get_fields() ?: [];

  // determine CPT slug (via key or literal)
  if (!empty($attributes['plxRelKey'])) {
    $cpt = get_theme_option($attributes['plxRelKey']);
  } elseif (!empty($attributes['plxRelSlug'])) {
    $cpt = $attributes['plxRelSlug'];
  } else {
    $cpt = '';
  }

  // build WP_Query args
  if (!empty($block_fields['posts'])) {
    $ids = array_map(fn($p) => is_object($p) ? $p->ID : (int) $p, $block_fields['posts']);
    $args = [
      'post__in' => $ids,
      'orderby' => 'post__in',
      'posts_per_page' => count($ids),
      'post_status' => 'publish',
    ];
    if ($cpt) {
      $args['post_type'] = $cpt;
    }
  } elseif ($cpt) {
    $args = [
      'post_type' => $cpt,
      'post_status' => 'publish',
      'posts_per_page' => -1,
      'orderby' => 'menu_order',
      'order' => 'ASC',
    ];
  } else {
    $args = [];
  }

  // fetch via timber
  $query = $args ? Timber::get_posts($args) : null;

  // extract the actual posts as an array
  $posts = $query ? (method_exists($query, 'to_array') ? $query->to_array() : (array) $query) : [];

  // decorate each post with its ACF subfields
  foreach ($posts as $post) {
    $post->fields = get_fields($post->ID) ?: [];
  }

  // render
  $context = Timber::context();
  $context['attributes'] = $attributes;
  $context['fields'] = $block_fields;
  $context['is_preview'] = $is_preview;
  $context['block_slug'] = $slug;
  $context['rel_posts'] = $posts;
  $context['content'] = $content;

  Timber::render("blocks/{$slug}/block.twig", $context);
}

// disable built in wp blocks
function plexible_allowed_blocks($allowed_block_types, $editor_context) {
  $dir = get_stylesheet_directory() . '/blocks';
  $blocks = [];

  if (is_dir($dir)) {
    foreach (new DirectoryIterator($dir) as $item) {
      if ($item->isDot() || !$item->isDir()) {
        continue;
      }
      $json = $item->getPathname() . '/block.json';
      if (file_exists($json)) {
        $data = json_decode(file_get_contents($json), true);
        if (!empty($data['name'])) {
          $blocks[] = $data['name'];
        }
      }
    }
  }

  return $blocks;
}
add_filter('allowed_block_types_all', 'plexible_allowed_blocks', 10, 2);

// disable gutenberg block editor on specific pages/post types
function plexible_block_editor($use_block_editor, $post_type) {
  $disabled = ['testimonials'];

  if (in_array($post_type, $disabled, true)) {
    return false;
  }
  return $use_block_editor;
}
add_filter('use_block_editor_for_post_type', 'plexible_block_editor', 10, 2);
