<?php

/**
 * Get Figure Fields
 *
 * Note: These fields must be wrapped in a group or repeater
 *
 * @param string $key Unique key for field repetition
 * @param array $include Providers to include, empty array includes ALL
 * @return array
 */

function get_figure_fields(string $key = '', array $include = [], string $parent_repeater = '', int $required = 1): array {
  $all_provider_choices = [
    'local-image' => 'Local Image',
    'local-video' => 'Local Video',
    'vimeo' => 'Vimeo',
    'youtube' => 'YouTube',
    'google-drive' => 'Google Drive',
    'lottie' => 'Lottie',
  ];

  /**
   * If no include list is passed, default to ALL providers
   */
  if (empty($include)) {
    $include = array_keys($all_provider_choices);
  }

  /**
   * Build provider field, filtering choices to only those in $include
   *
   * Note: Provider must always be present as the first field
   */
  $filtered_choices = [];
  foreach ($include as $slug) {
    if (isset($all_provider_choices[$slug])) {
      $filtered_choices[$slug] = $all_provider_choices[$slug];
    }
  }

  $fields = [
    [
      'key' => "field_figure_provider_{$key}",
      'label' => 'Image / Video Source',
      'name' => 'provider',
      'type' => 'select',
      'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
      'required' => $required,
      'choices' => $filtered_choices,
      'default_value' => array_key_exists('local-image', $filtered_choices)
        ? 'local-image'
        : array_key_first($filtered_choices),
    ],
  ];

  $all_fields = [
    'local-image' => [
      [
        'key' => "field_src_local_image_{$key}",
        'label' => 'Local Image Embed',
        'name' => 'src_local_image',
        'type' => 'image',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'return_format' => 'url',
        'library' => 'all',
        'mime_types' => ACF_CONSTANTS['ACCEPTED_IMAGE_FILETYPES'],
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'local-image',
            ],
          ],
        ],
      ],
    ],

    'local-video' => [
      [
        'key' => "field_src_local_video_{$key}",
        'label' => 'Local Video Embed',
        'name' => 'src_local_video',
        'type' => 'file',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'return_format' => 'url',
        'mime_types' => ACF_CONSTANTS['ACCEPTED_VIDEO_FILETYPES'],
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'local-video',
            ],
          ],
        ],
      ],
    ],

    'vimeo' => [
      [
        'key' => "field_src_vimeo_{$key}",
        'label' => 'Vimeo Embed',
        'name' => 'src_vimeo',
        'type' => 'number',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'vimeo',
            ],
          ],
        ],
      ],
    ],

    'youtube' => [
      [
        'key' => "field_src_youtube_{$key}",
        'label' => 'Youtube Embed',
        'name' => 'src_youtube',
        'type' => 'text',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'youtube',
            ],
          ],
        ],
      ],
    ],

    'google-drive' => [
      [
        'key' => "field_src_google_drive_{$key}",
        'label' => 'Google Drive',
        'name' => 'src_google_drive',
        'type' => 'text',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'google-drive',
            ],
          ],
        ],
      ],
    ],

    'lottie' => [
      [
        'key' => "field_src_lottie_{$key}",
        'label' => 'Lottie Embed',
        'name' => 'src_lottie',
        'type' => 'text',
        'wpml_cf_preferences' => ACF_CONSTANTS['TRANSLATE_COPY_ONCE'],
        'required' => $required,
        'conditional_logic' => [
          [
            [
              'field' => "field_figure_provider_{$key}",
              'operator' => '==',
              'value' => 'lottie',
            ],
          ],
        ],
      ],
    ],
  ];

  /**
   * Append only the blocks asked for in $include
   */
  foreach ($include as $slug) {
    if (isset($all_fields[$slug])) {
      $fields = array_merge($fields, $all_fields[$slug]);
    }
  }

  /**
   * If parent_repeater is provided, inject it on every field
   */
  if ($parent_repeater !== '') {
    foreach ($fields as &$field) {
      $field['parent_repeater'] = $parent_repeater;
    }
    unset($field);
  }

  return $fields;
}
