import { CUSTOM_EVENT } from '../config';
import { $mouse } from '../stores/mouse.js';
import { $body } from '../utils/dom.js';
import { delegateEvent } from '../utils/events.js';

export default class Cursor {
  constructor() {
    /* Binding */
    this.onMouseEnterBind = this.onMouseEnter.bind(this);
    this.onMouseLeaveBind = this.onMouseLeave.bind(this);
    this.onMouseMoveBind = this.onMouseMove.bind(this);
    this.onClickBind = this.onClick.bind(this);

    /* UI */
    this.$cursor = document.querySelector('[data-cursor-root]');
    if (!this.$cursor) return;

    /* Data */
    this.type = '';

    /* Call */
    this.init();
  }

  init() {
    this.bindEvents();
  }

  bindEvents() {
    delegateEvent($body, 'mouseenter', '[data-cursor]', this.onMouseEnterBind, true);
    delegateEvent($body, 'mouseleave', '[data-cursor]', this.onMouseLeaveBind, true);
    delegateEvent($body, 'click', '[data-cursor]', this.onClickBind);
    window.addEventListener(CUSTOM_EVENT.CURSOR_RESET, this.onMouseLeaveBind);
    $mouse.subscribe(this.onMouseMoveBind);
  }

  onMouseEnter(e) {
    const $target = e.target;
    this.type = $target.dataset.cursor;
    this.setCursorType();
  }

  onMouseLeave(e) {
    this.type = '';
    this.setCursorType();
  }

  onMouseMove(values) {
    this.$cursor.style.transform = `translate3d(${values.smoothX}px, ${values.smoothY}px, 0)`;
  }

  onClick() {
    this.type = '';
    this.setCursorType();
  }

  setCursorType() {
    if (this.type) {
      this.$cursor.classList.add('is-active');
    } else {
      this.$cursor.classList.remove('is-active');
    }
    this.$cursor.dataset.state = this.type;
  }
}
