/**
 * Click handler for term filters
 * @param {MouseEvent} event
 */
import { initCube3FilterSlider } from './swiper.js';

async function handleTermFilterClick(event) {
  event.preventDefault();
  const btn = event.currentTarget;

  let {
    ajaxActionName,
    ajaxContainer,
    ajaxUrl,
    ajaxCallback = '',
    ajaxQueryArgs = '{}',
    ajaxNonce,
  } = btn.dataset;

  const isAlreadyActive = btn.dataset.active === 'true';
  const termId = isAlreadyActive ? 'all' : btn.dataset.filterId;

  const layoutRow = btn.closest('.layout-row');
  if (layoutRow) {
    const loadMoreBtn = layoutRow.querySelector('[data-js-target="pagination.loadMore"]');
    if (loadMoreBtn) {
      loadMoreBtn.dataset.ajaxTermId = termId;
    }
  }

  const container = document.querySelector(`.${ajaxContainer}`);

  if (!container) {
    console.error('Missing content container for filtered posts.');
    return;
  }

  const allButtons = btn.parentNode.querySelectorAll('.filter');
  allButtons.forEach((b) => (b.dataset.active = 'false'));

  if (isAlreadyActive) {
    // If the same filter clicked again, activate the "All" filter button
    const allBtn = btn.parentNode.querySelector('[data-filter-id="all"]');
    if (allBtn) {
      allBtn.dataset.active = 'true';
    }
  } else {
    // Normal case: set clicked button as active
    btn.dataset.active = 'true';
  }

  try {
    container.style.opacity = '0.5';

    const payload = new URLSearchParams({
      action: ajaxActionName,
      security: ajaxNonce,
      term_id: termId,
      loader_callback: ajaxCallback,
      query_args: JSON.stringify(JSON.parse(ajaxQueryArgs)),
    });

    const res = await fetch(ajaxUrl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: payload.toString(),
    });

    const text = await res.text();
    let data;

    try {
      data = JSON.parse(text);
    } catch {
      console.error('Invalid JSON returned:', text);
      throw new Error('Invalid JSON from server');
    }

    if (!data.success) {
      throw new Error(data.data || 'Unknown AJAX error');
    }

    container.innerHTML =
      (data.data.html || '').trim() || '<p>No posts found.</p>';
    initCube3FilterSlider();
    container.style.opacity = '1';

    //reset Load More button
    const layoutRow = btn.closest('.layout-row');
    if (layoutRow) {
      const loadMoreBtn = layoutRow.querySelector('[data-js-target="pagination.loadMore"]');

      if (loadMoreBtn) {
        loadMoreBtn.dataset.page = '1';
        loadMoreBtn.dataset.ajaxTermId = termId;

        if(data.data.max_pages < 2){
          loadMoreBtn.disabled = true;
          loadMoreBtn.style.display = 'none';
        }else{
          loadMoreBtn.disabled = false;
          loadMoreBtn.style.display = '';
        }

      }
    }


  } catch (err) {
    console.error('Term filter failed:', err);
    container.innerHTML =
      '<p class="error-message">Error loading posts. Try again.</p>';
  }
}

/**
 * Init term filter buttons
 */
function initTermFilterButtons() {
  const buttons = document.querySelectorAll(
    '[data-js-target="filter.termFilter"]'
  );
  buttons.forEach((btn) =>
    btn.addEventListener('click', handleTermFilterClick)
  );
}

document.addEventListener('DOMContentLoaded', initTermFilterButtons);
