import { gsap } from 'gsap';
import { SplitText } from 'gsap/SplitText';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

export class SplitLinesScrollEffect {
  constructor(element) {
    if (!element || !(element instanceof HTMLElement)) {
      throw new Error('Invalid element provided to SplitLineScrollEffect');
    }

    this.element = element;
    this.splitInstance = null;
    this.initializeEffect();
  }

  initializeEffect() {
    // Ensure visibility in case CSS is hiding it
    gsap.set(this.element, { visibility: 'visible' });

    // Find the trigger parent
    this.triggerEl = this.element.closest('.block[data-plx]');
    if (!this.triggerEl) {
      console.warn(
        `No parent section[data-plx] found for splitLine element:`,
        this.element,
        `Skipping animation.`
      );
      return;
    }

    // Bind resize to re-create triggers
    window.addEventListener('resize', this.refresh);

    // Create the ScrollTrigger
    this.createTrigger();
  }

  createTrigger() {
    this.trigger = ScrollTrigger.create({
      trigger: this.triggerEl,
      start: 'top 85%',
      end: 'bottom 15%',
      onEnter: () => this.animateIn(),
      onLeave: () => this.animateOut(),
      onEnterBack: () => this.animateIn(),
      onLeaveBack: () => this.animateOut(),
    });
  }

  animateIn() {
    // Revert any prior split
    if (this.splitInstance?.revert) {
      this.splitInstance.revert();
    }

    // Don’t run on empty text
    if (!this.element.textContent.trim()) {
      return;
    }

    try {
      // Split into lines
      this.splitInstance = new SplitText(this.element, {
        type: 'lines',
        mask: 'lines',
      });

      const lines = this.splitInstance.lines || [];
      if (lines.length) {
        gsap.from(lines, {
          y: '100%',
          duration: 0.8,
          ease: 'power2.out',
          stagger: 0.15,
        });
      } else {
        // Nothing to animate? revert immediately
        this.splitInstance.revert();
      }
    } catch (e) {
      console.error('SplitText error for', this.element, e);
      this.splitInstance?.revert();
    }
  }

  animateOut() {
    if (this.splitInstance?.revert) {
      this.splitInstance.revert();
    }
  }

  kill() {
    this.trigger?.kill();
    this.splitInstance?.revert();
    window.removeEventListener('resize', this.resizeHandler);
  }

  refresh() {
    this.kill();
    this.initializeEffect();
  }
}
