<?php
/**
 * Plugin Name: Simple WebP Converter
 * Plugin URI: https://ignatovdesigns.com/simple-webp-converter
 * Description: Automatically converts uploaded images to WebP format for better performance and smaller file sizes.
 * Version: 1.0.1
 * Author: Krasimir Ignatov
 * Author URI: https://ignatovdesigns.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simple-webp-converter
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SIMPWEBPCONV_VERSION', '1.0.0');
define('SIMPWEBPCONV_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SIMPWEBPCONV_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SIMPWEBPCONV_PLUGIN_FILE', __FILE__);

/**
 * Main plugin class
 */
class SimpleWebPConverter {
    
    private $options;
    
    public function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // Initialize plugin
        $this->options = get_option('simpwebpconv_options', array(
            'quality' => 80
        ));
        
        // Initialize stats if they don't exist
        if (!get_option('simpwebpconv_stats')) {
            add_option('simpwebpconv_stats', array(
                'total_converted' => 0,
                'bytes_saved' => 0,
                'original_size' => 0,
                'webp_size' => 0
            ));
        }
        
        // Add hooks
        add_filter('wp_handle_upload', array($this, 'convert_uploaded_image'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('wp_ajax_simpwebpconv_convert_existing', array($this, 'handle_bulk_convert'));
    }
    
    /**
     * Convert uploaded image to WebP
     */
    public function convert_uploaded_image($upload) {
        if (!isset($upload['file']) || !isset($upload['type'])) {
            return $upload;
        }
        
        $file_path = $upload['file'];
        $file_type = $upload['type'];
        
        // Check if it's an image we can convert
        $supported_types = array('image/jpeg', 'image/png', 'image/gif');
        if (!in_array($file_type, $supported_types)) {
            return $upload;
        }
        
        // Convert to WebP
        $webp_path = $this->convert_to_webp($file_path, $file_type);
        
        if ($webp_path && file_exists($webp_path)) {
            // Get file sizes for statistics
            $original_size = filesize($file_path);
            $webp_size = filesize($webp_path);
            $bytes_saved = $original_size - $webp_size;
            
            // Update statistics
            $this->update_stats($original_size, $webp_size, $bytes_saved);
            
            // Update upload info to point to WebP file
            $upload['file'] = $webp_path;
            $upload['url'] = str_replace(basename($upload['url']), basename($webp_path), $upload['url']);
            $upload['type'] = 'image/webp';
            
            // Always delete original file using WordPress function
            wp_delete_file($file_path);
        }
        
        return $upload;
    }
    
    /**
     * Convert image to WebP format
     */
    private function convert_to_webp($file_path, $file_type) {
        if (!extension_loaded('gd')) {
            return false;
        }
        
        $image = null;
        
        // Create image resource based on type
        switch ($file_type) {
            case 'image/jpeg':
                $image = imagecreatefromjpeg($file_path);
                break;
            case 'image/png':
                $image = imagecreatefrompng($file_path);
                // Preserve transparency
                imagealphablending($image, false);
                imagesavealpha($image, true);
                break;
            case 'image/gif':
                $image = imagecreatefromgif($file_path);
                break;
        }
        
        if (!$image) {
            return false;
        }
        
        // Generate WebP file path
        $webp_path = preg_replace('/\.[^.]+$/', '.webp', $file_path);
        
        // Convert to WebP
        $result = imagewebp($image, $webp_path, $this->options['quality']);
        imagedestroy($image);
        
        return $result ? $webp_path : false;
    }
    
    /**
     * Update conversion statistics
     */
    private function update_stats($original_size, $webp_size, $bytes_saved) {
        $stats = get_option('simpwebpconv_stats', array(
            'total_converted' => 0,
            'bytes_saved' => 0,
            'original_size' => 0,
            'webp_size' => 0
        ));
        
        $stats['total_converted']++;
        $stats['bytes_saved'] += max(0, $bytes_saved); // Ensure we don't subtract if WebP is larger
        $stats['original_size'] += $original_size;
        $stats['webp_size'] += $webp_size;
        
        update_option('simpwebpconv_stats', $stats);
    }
    
    /**
     * Update bulk conversion statistics
     */
    private function update_bulk_stats($count, $original_size, $webp_size, $bytes_saved) {
        $stats = get_option('simpwebpconv_stats', array(
            'total_converted' => 0,
            'bytes_saved' => 0,
            'original_size' => 0,
            'webp_size' => 0
        ));
        
        $stats['total_converted'] += $count; // Add the actual number of converted images
        $stats['bytes_saved'] += max(0, $bytes_saved);
        $stats['original_size'] += $original_size;
        $stats['webp_size'] += $webp_size;
        
        update_option('simpwebpconv_stats', $stats);
    }
    
    /**
     * Format bytes into human readable format
     */
    private function format_bytes($size, $precision = 2) {
        if ($size === 0) return '0 B';
        
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $base = log($size, 1024);
        
        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $units[floor($base)];
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'settings_page_simple-webp-converter') {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script(
            'simpwebpconv-admin',
            SIMPWEBPCONV_PLUGIN_URL . 'admin.js',
            array('jquery'),
            SIMPWEBPCONV_VERSION,
            true
        );
        
        // Localize script with AJAX data
        wp_localize_script('simpwebpconv-admin', 'simpwebpconv_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('simpwebpconv_bulk_convert'),
            'converting' => __('Converting...', 'simple-webp-converter')
        ));
        
        // Add inline CSS
        $css = "
        .card {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 20px;
            margin: 20px 0;
        }
        .swc-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        .swc-stat-card {
            background: linear-gradient(135deg, #30B881 0%, #038D92 100%);
            color: white;
            padding: 30px 20px;
            border-radius: 8px;
            text-align: center;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .swc-stat-number {
            font-size: 2.5em;
            font-weight: bold;
            line-height: 1;
            margin-bottom: 10px;
        }
        .swc-stat-label {
            font-size: 0.9em;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        .progress-bar {
            width: 100%;
            height: 20px;
            background: #f0f0f0;
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 10px;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(135deg, #30B881 0%, #038D92 100%);
            transition: width 0.3s ease;
            width: 0%;
        }
        ";
        
        wp_add_inline_style('wp-admin', $css);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            esc_html__('Simple WebP Converter', 'simple-webp-converter'),
            esc_html__('WebP Converter', 'simple-webp-converter'),
            'manage_options',
            'simple-webp-converter',
            array($this, 'admin_page')
        );
        
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Initialize admin settings
     */
    public function admin_init() {
        register_setting('simpwebpconv_options', 'simpwebpconv_options', array($this, 'sanitize_options'));
        
        add_settings_section(
            'simpwebpconv_main_section',
            esc_html__('Conversion Settings', 'simple-webp-converter'),
            null,
            'simple-webp-converter'
        );
        
        add_settings_field(
            'quality',
            esc_html__('WebP Quality', 'simple-webp-converter'),
            array($this, 'quality_field'),
            'simple-webp-converter',
            'simpwebpconv_main_section'
        );
    }
    
    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = array();
        $sanitized['quality'] = max(1, min(100, intval($input['quality'])));
        return $sanitized;
    }
    
    /**
     * Quality field callback
     */
    public function quality_field() {
        $quality = isset($this->options['quality']) ? $this->options['quality'] : 80;
        echo '<input type="number" name="simpwebpconv_options[quality]" value="' . esc_attr($quality) . '" min="1" max="100" />';
        echo '<p class="description">' . esc_html__('Quality of WebP images (1-100). Higher values mean better quality but larger file sizes.', 'simple-webp-converter') . '</p>';
    }
    
    /**
     * Handle AJAX request for bulk conversion
     */
    public function handle_bulk_convert() {
        // Check nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'simpwebpconv_bulk_convert')) {
            wp_send_json_error('Security check failed');
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }
        
        // Get all image attachments that aren't WebP
        $args = array(
            'post_type' => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/png', 'image/gif'),
            'post_status' => 'inherit',
            'posts_per_page' => -1
        );
        
        $attachments = get_posts($args);
        $converted_count = 0;
        $total_original_size = 0;
        $total_webp_size = 0;
        
        foreach ($attachments as $attachment) {
            $file_path = get_attached_file($attachment->ID);
            
            if (!file_exists($file_path)) {
                continue;
            }
            
            $file_type = get_post_mime_type($attachment->ID);
            $original_size = filesize($file_path);
            
            // Convert to WebP
            $webp_path = $this->convert_to_webp($file_path, $file_type);
            
            if ($webp_path && file_exists($webp_path)) {
                $webp_size = filesize($webp_path);
                
                // Update attachment metadata
                update_attached_file($attachment->ID, $webp_path);
                wp_update_post(array(
                    'ID' => $attachment->ID,
                    'post_mime_type' => 'image/webp'
                ));
                
                // Update attachment metadata for WordPress
                $metadata = wp_generate_attachment_metadata($attachment->ID, $webp_path);
                wp_update_attachment_metadata($attachment->ID, $metadata);
                
                // Delete original file
                wp_delete_file($file_path);
                
                $converted_count++;
                $total_original_size += $original_size;
                $total_webp_size += $webp_size;
            }
        }
        
        // Update bulk conversion statistics
        if ($converted_count > 0) {
            $bytes_saved = $total_original_size - $total_webp_size;
            $this->update_bulk_stats($converted_count, $total_original_size, $total_webp_size, $bytes_saved);
        }
        
        wp_send_json_success(array(
            'message' => sprintf(
                /* translators: %d: number of images converted */
                esc_html__('Successfully converted %d images to WebP format!', 'simple-webp-converter'),
                $converted_count
            )
        ));
    }
    
    /**
     * Admin page
     */
    public function admin_page() {
        if (!extension_loaded('gd')) {
            echo '<div class="notice notice-error"><p>' . esc_html__('GD extension is required for image conversion.', 'simple-webp-converter') . '</p></div>';
        }
        
        // Get statistics
        $stats = get_option('simpwebpconv_stats', array(
            'total_converted' => 0,
            'bytes_saved' => 0,
            'original_size' => 0,
            'webp_size' => 0
        ));
        
        $savings_percentage = $stats['original_size'] > 0 ? 
            round(($stats['bytes_saved'] / $stats['original_size']) * 100, 1) : 0;
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Simple WebP Converter', 'simple-webp-converter'); ?></h1>
            
            <!-- Statistics Dashboard -->
            <div class="swc-stats-grid">
                <div class="swc-stat-card">
                    <div class="swc-stat-number"><?php echo number_format($stats['total_converted']); ?></div>
                    <div class="swc-stat-label"><?php echo esc_html__('Images Converted', 'simple-webp-converter'); ?></div>
                </div>
                <div class="swc-stat-card">
                    <div class="swc-stat-number"><?php echo esc_html($this->format_bytes($stats['bytes_saved'])); ?></div>
                    <div class="swc-stat-label"><?php echo esc_html__('Space Saved', 'simple-webp-converter'); ?></div>
                </div>
                <div class="swc-stat-card">
                    <div class="swc-stat-number"><?php echo esc_html($savings_percentage); ?>%</div>
                    <div class="swc-stat-label"><?php echo esc_html__('Size Reduction', 'simple-webp-converter'); ?></div>
                </div>
            </div>
            
            <!-- Bulk Convert Section -->
            <div class="card">
                <h2><?php echo esc_html__('Bulk Convert Existing Images', 'simple-webp-converter'); ?></h2>
                <p><?php echo esc_html__('Convert all existing JPEG, PNG, and GIF images in your media library to WebP format.', 'simple-webp-converter'); ?></p>
                <button id="convert-existing" class="button button-primary"><?php echo esc_html__('Convert Existing Images', 'simple-webp-converter'); ?></button>
                <div id="conversion-progress" style="display: none; margin-top: 15px;">
                    <div class="progress-bar">
                        <div class="progress-fill"></div>
                    </div>
                    <p><?php echo esc_html__('Converting images, please wait...', 'simple-webp-converter'); ?></p>
                </div>
            </div>
            
            <div class="card">
                <h2><?php echo esc_html__('How it works', 'simple-webp-converter'); ?></h2>
                <p><?php echo esc_html__('This plugin automatically converts uploaded images (JPEG, PNG, GIF) to WebP format for better performance and smaller file sizes.', 'simple-webp-converter'); ?></p>
                <ul>
                    <li><?php echo esc_html__('Upload any image - it will be automatically converted to WebP', 'simple-webp-converter'); ?></li>
                    <li><?php echo esc_html__('Original files are automatically deleted after conversion', 'simple-webp-converter'); ?></li>
                    <li><?php echo esc_html__('Adjust quality settings below to control file size vs quality', 'simple-webp-converter'); ?></li>
                </ul>
            </div>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('simpwebpconv_options');
                do_settings_sections('simple-webp-converter');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }
    }

// Plugin activation hook
register_activation_hook(__FILE__, 'simpwebpconv_activate');
function simpwebpconv_activate() {
    // Check if GD extension is loaded
    if (!extension_loaded('gd')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(esc_html__('Simple WebP Converter requires the GD extension to be installed and enabled.', 'simple-webp-converter'));
    }
    
    // Set default options
    add_option('simpwebpconv_options', array(
        'quality' => 80
    ));
}

// Plugin deactivation hook
register_deactivation_hook(__FILE__, 'simpwebpconv_deactivate');
function simpwebpconv_deactivate() {
    // Clean up if needed
}

// Plugin uninstall hook
register_uninstall_hook(__FILE__, 'simpwebpconv_uninstall');
function simpwebpconv_uninstall() {
    delete_option('simpwebpconv_options');
    delete_option('simpwebpconv_stats'); // Clean up statistics on uninstall
}

// Initialize plugin
new SimpleWebPConverter();