<?php

// bail if someone tries to load this directly
if (!defined('ABSPATH')) {
  exit();
}

/**
 * Global load more posts
 */
add_action('wp_ajax_load_more_posts', 'load_more_posts_callback');
add_action('wp_ajax_nopriv_load_more_posts', 'load_more_posts_callback');

function load_more_posts_callback() {
  check_ajax_referer('load-more-posts', 'security');

  // pull & sanitize inbound params
  $component = sanitize_text_field($_POST['component'] ?? '');
  $loader_callback = sanitize_text_field($_POST['loader_callback'] ?? '');
  $page = max(1, intval($_POST['page'] ?? 1));
  $per_page = max(1, intval($_POST['per_page'] ?? 12));
  $query_args = json_decode(wp_unslash($_POST['query_args'] ?? '{}'), true) ?: [];
  $component_args = json_decode(wp_unslash($_POST['component_args'] ?? '{}'), true) ?: [];
  $term_id = isset($_POST['term_id']) ? intval(wp_unslash($_POST['term_id'])) : 0;
  $taxonomy = get_primary_taxonomy_for_post_type($query_args['post_type']);

  // enforce pagination + published
  $query_args = array_merge($query_args, [
    'post_status' => 'publish',
    'posts_per_page' => $per_page,
    'paged' => $page,
  ]);

  if (!empty($term_id) && $term_id !== 'all') {
    $query_args['tax_query'][] = [
      'taxonomy' => $taxonomy,
      'field' => 'term_id',
      'terms' => (array)$term_id,
    ];
  }

  $q = new WP_Query($query_args);
  $max = $q->max_num_pages;

  ob_start();
  if ($q->have_posts()) {
    while ($q->have_posts()) {
      $q->the_post();
      // if a custom loader exists, call it
      if ($loader_callback && function_exists($loader_callback)) {
        $out = call_user_func($loader_callback, get_post(), $component_args);
        if (is_string($out)) {
          echo $out;
        }
      }
      // otherwise fall back to the standard component()
      elseif ($component) {
        component($component, $component_args);
      }
    }
    wp_reset_postdata();
  }
  $html = ob_get_clean();

  wp_send_json_success([
    'html' => $html,
    'max_pages' => $max,
  ]);
}

/**
 * Specific callbacks
 */
function blog_loader(WP_Post $post, array $base_args) {
  setup_postdata($post);

  $args = array_merge($base_args, [
    'title' => get_the_title($post),
    'link' => get_the_permalink($post),
    'date' => get_localized_date($post->ID),
    'feat_image' => get_feat_img($post->ID),
    'author_image' => get_field('user_image', 'user_' . $post->post_author),
    'author_name' => get_author_full_name($post->post_author),
  ]);

  return component('blog/blog-card', $args);
}
