<?php

/**
 * Builder
 * Serves as the main entry point for `Builder` ACF flexible content.
 * @see `lib/acf/schemas/field-groups/builder.php`
 */

$counters = [];
$modules = get_theme_option('modules');

/**
 * Generate block specific id, classes and inline styles
 */
function generate_block_options() {
  $styles = [];
  $classes = [];
  $anchor_id = '';

  // inline variables
  if ($spacing = get_sub_field('opt_section_space')) {
    $styles[] = '--multiplier-space: ' . esc_attr($spacing);
  }
  if ($space_top = get_sub_field('opt_space_top')) {
    $styles[] = '--multiplier-space-top: ' . esc_attr($space_top);
  }
  if ($space_bottom = get_sub_field('opt_space_bottom')) {
    $styles[] = '--multiplier-space-bottom: ' . esc_attr($space_bottom);
  }
  if ($text_color = get_sub_field('opt_text_color')) {
    $styles[] = '--text-color: ' . esc_attr($text_color);
  }
  if ($background_color = get_sub_field('opt_background_color')) {
    $styles[] = '--background-color: ' . esc_attr($background_color);
  }
  if ($border_color = get_sub_field('opt_border_color')) {
    $styles[] = '--border-color: ' . esc_attr($border_color);
  }
  if ($card_color = get_sub_field('opt_card_color')) {
    $styles[] = '--card-color: ' . esc_attr($card_color);
  }
  if ($grid_size = get_sub_field('opt_grid_size')) {
    $styles[] = '--grid-size: ' . esc_attr($grid_size);
  }
  if ($background_image = get_sub_field('opt_background_image')) {
    $styles[] = '--background-image: url(' . esc_attr($background_image) . ')';
    $classes[] = esc_attr('has-background-image');
  }
  if ($v_background_image = get_sub_field('var_background_image')) {
    $styles[] = '--background-image: url(' . esc_attr($v_background_image) . ')';
    $classes[] = esc_attr('has-background-image');
  }

  // classes
  if ($opt_theme = get_sub_field('opt_theme')) {
    $theme = match ($opt_theme ?? null) {
      '1' => 'theme--light',
      '2' => 'theme--dark',
      '3' => 'theme--system',
      default => null,
    };
    $classes[] = esc_attr($theme);
  }
  if ($section_border = get_sub_field('opt_section_border') || get_sub_field('opt_border')) {
    $classes[] = esc_attr('has-border');
  }
  if ($section_overlay = get_sub_field('opt_background_overlay')) {
    $classes[] = esc_attr('has-overlay');
  }

  // id
  if ($anchor = get_sub_field('opt_anchor')) {
    $anchor_id = htmlspecialchars($anchor);
  }

  $style_string = !empty($styles) ? 'style="' . implode('; ', $styles) . '"' : '';
  $class_string = !empty($classes) ? implode(' ', $classes) : '';
  $id_string = !empty($anchor_id) ? 'id="' . $anchor_id . '"' : '';

  return [$style_string, $class_string, $id_string];
}

/**
 * Match acf row layout based on provided index
 */
function is_layout_match($block_index_key) {
  $layout = get_row_layout();
  return $layout == BLOCK_INDEX[$block_index_key]['name'] || $layout == BLOCK_INDEX[$block_index_key]['legacy_name'];
}
?>

<div class="plx-builder">
  <?php if (have_rows('cube_page_builder')): ?>
    <?php while (have_rows('cube_page_builder')): ?>
        <?php the_row(); ?>


        <?php if (is_layout_match('0')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $type = null;
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'wysiwyg' => get_sub_field('content'),
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="0"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-00/plx-00' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('1')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--split',
            '2' => 'variant--blox',
            '3' => 'variant--parallax-grid',
            '4' => 'variant--lower-third',
            default => null,
          };
          $template_data = [
            'slides' => get_sub_field('slides') ?: [],
            'hero' => get_sub_field('hero') ?: [],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="1"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-01/plx-01' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('2')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $post_type_slug = get_theme_option('custom_post_type_1_slug');
          $filtered_posts = get_sub_field('posts');
          $args = [
            'post_type' => $post_type_slug,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'menu_order',
            'order' => 'ASC',
          ];
          if (!empty($filtered_posts)) {
            $post_ids = array_map(function ($post) {
              return $post->ID;
            }, $filtered_posts);
            $args['post__in'] = $post_ids;
            $args['orderby'] = 'post__in'; // keep post order from relationship field
          }
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--grid',
            default => null,
          };
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'align' => $align,
            'query_args' => $args,
            'type' => $type,
            'custom' => [
              'card_text' => get_sub_field('var_card_text_color'),
              'card_background' => get_sub_field('var_card_background_color'),
              'card_border' => get_sub_field('var_card_border_color'),
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="2"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-02/plx-02' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('3')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $post_type_slug = get_theme_option('custom_post_type_2_slug');
          $filtered_posts = get_sub_field('posts');
          $args = [
            'post_type' => $post_type_slug,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'menu_order',
            'order' => 'ASC',
          ];
          if (!empty($filtered_posts)) {
            $post_ids = array_map(function ($post) {
              return $post->ID;
            }, $filtered_posts);
            $args['post__in'] = $post_ids;
            $args['orderby'] = 'post__in'; // keep post order from relationship field
          }
          $terms = get_terms([
            'taxonomy' => 'custom_post_type_2_category',
            'hide_empty' => true,
          ]);
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--slider',
            '2' => 'variant--grid',
            '3' => 'variant--carousel',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
            ],
            'query_args' => $args,
            'terms' => $terms,
            'type' => $type,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="3"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-03/plx-03' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('4')): ?>
            <?php
            [$styles, $classes, $id] = generate_block_options();
            $post_type_slug = get_theme_option('custom_post_type_3_slug');
            $filtered_posts = get_sub_field('posts');
            $args = [
              'post_type' => $post_type_slug,
              'posts_per_page' => -1,
              'post_status' => 'publish',
              'orderby' => 'menu_order',
              'order' => 'ASC',
            ];
            if (!empty($filtered_posts)) {
              $post_ids = array_map(function ($post) {
                return $post->ID;
              }, $filtered_posts);
              $args['post__in'] = $post_ids;
              $args['orderby'] = 'post__in'; // keep post order from relationship field
            }
            $type = null;
            $template_data = [
              'content' => [
                'has_caption' => get_sub_field('has_caption'),
                'caption' => get_sub_field('caption'),
                'has_title' => get_sub_field('has_title'),
                'title' => get_sub_field('title'),
                'has_body' => get_sub_field('has_body'),
                'body' => get_sub_field('body'),
                'has_footer' => get_sub_field('has_footer'),
                'button_group' => get_sub_field('button_group') ?: [],
              ],
              'query_args' => $args,
            ];
            ?>
            <section
              <?= $id ?>
              data-plx="4"
              class="<?= cn('section', $classes) ?>"
              <?= $styles ?>
            >
              <?php component('blocks/plx-04/plx-04' . ($type ? '-' . $type : ''), $template_data); ?>
            </section>
        <?php endif; ?>


        <?php if (is_layout_match('5')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--slider',
            '2' => 'variant--card',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'gallery' => get_sub_field('gallery') ?: [],
            'align' => $align,
            'type' => $type,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="5"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-05/plx-05' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('6')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          function generate_icon_options() {
            $styles = [];
            $classes = [];
            if ($icon_color = get_sub_field('opt_icon_color')) {
              $styles[] = '--icon-color: ' . esc_attr($icon_color);
            }
            if ($icon_background_color = get_sub_field('opt_icon_background_color')) {
              $styles[] = '--icon-background-color: ' . esc_attr($icon_background_color);
            }
            if ($icon_gradient_color = get_sub_field('opt_icon_gradient_color')) {
              $styles[] = '--icon-gradient-color: ' . esc_attr($icon_gradient_color);
            }
            if ($icon_style = get_sub_field('opt_icon_style')) {
              $classes[] = esc_attr('icon--style-' . $icon_style);
            }
            $style_string = !empty($styles) ? 'style="' . implode('; ', $styles) . '"' : '';
            $class_string = !empty($classes) ? implode(' ', $classes) : '';
            return [$style_string, $class_string];
          }
          [$icon_styles, $icon_classes] = generate_icon_options();
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--contact',
            '2' => 'variant--blox',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'form' => [
              'id' => get_sub_field('form_id'),
            ],
            'contact' => [
              'email_label' => get_field('contact_email_label', 'options'),
              'email' => get_field('contact_email', 'options'),
              'phone_label' => get_field('contact_phone_label', 'options'),
              'phone' => get_field('contact_phone', 'options'),
              'address_label' => get_field('contact_address_label', 'options'),
              'address' => get_field('contact_address', 'options'),
              'icon_styles' => $icon_styles,
              'icon_classes' => $icon_classes,
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="6"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-06/plx-06' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('7')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $var_column_breakpoint = get_sub_field('var_column_breakpoint');
          $column_breakpoint = !empty($var_column_breakpoint)
            ? '--column-breakpoint: ' . $var_column_breakpoint . 'px'
            : '';
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--two',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'items' => get_sub_field('item') ?: [],
            'align' => $align,
            'column_breakpoint' => $column_breakpoint,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="7"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-07/plx-07' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('8')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
        $type = match (get_sub_field('var_type') ?? null) {
          '1' => 'variant--light',
          '2' => 'variant--dark',
          default => null,
        };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
            ],
            'items' => get_sub_field('item') ?: [],
            'map' => [
              'location' => get_sub_field('map'),
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="8"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-08/plx-08' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('9')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $item_align = match (get_sub_field('var_item_align') ?? null) {
            '1' => 'align--center',
            '2' => 'align--end',
            default => null,
          };
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--blox',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'image' => get_sub_field('image'),
            'figure' => get_sub_field('figure'),
            'align' => $align,
            'item_align' => $item_align,
            'item_reverse' => get_sub_field('var_item_reverse') ? 'reverse' : null,
            'items' => get_sub_field('item') ?: [],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="9"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-09/plx-09' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('10')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'gallery' => get_sub_field('gallery'),
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="10"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-10/plx-10' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('11')): ?>
            <?php
            [$styles, $classes, $id] = generate_block_options();
            $type = null;
            $template_data = [
              'content' => [
                'has_caption' => get_sub_field('has_caption'),
                'caption' => get_sub_field('caption'),
                'has_title' => get_sub_field('has_title'),
                'title' => get_sub_field('title'),
                'has_body' => get_sub_field('has_body'),
                'body' => get_sub_field('body'),
              ],
            ];
            ?>
            <section
              <?= $id ?>
              data-plx="11"
              class="<?= cn('section', $classes) ?>"
              <?= $styles ?>
            >
              <?php component('blocks/plx-11/plx-11' . ($type ? '-' . $type : ''), $template_data); ?>
            </section>
        <?php endif; ?>


        <?php if (is_layout_match('12')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'image' => get_sub_field('image'),
            'items' => get_sub_field('item') ?: [],
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="12"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-12/plx-12' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('13') && $modules['testimonial']['enable_testimonials']): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $post_type_slug = MODULES['testimonial']['slug'];
          $filtered_posts = get_sub_field('posts');
          $args = [
            'post_type' => $post_type_slug,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'menu_order',
            'order' => 'ASC',
          ];
          if (!empty($filtered_posts)) {
            $post_ids = array_map(function ($post) {
              return $post->ID;
            }, $filtered_posts);
            $args['post__in'] = $post_ids;
            $args['orderby'] = 'post__in'; // keep post order from relationship field
          }
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--slider',
            '2' => 'variant--blox',
            default => null,
          };
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'image' => get_sub_field('image') ?: [],
            'query_args' => $args,
            'type' => $type,
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="13"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-13/plx-13' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('14')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="14"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-14/plx-14' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('15')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--full-width',
            default => null,
          };
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'figure' => get_sub_field('figure') ?: [],
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="15"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-15/plx-15' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('16')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $list_size = match (get_sub_field('var_list_size') ?? null) {
            '1' => 'small',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'list' => get_sub_field('list') ?: [],
            'custom' => [
              'list_size' => $list_size,
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="16"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-16/plx-16' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('17')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--slider',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'cards' => get_sub_field('cards') ?: [],
            'align' => $align,
            'icon' => [
              'has_background' => get_sub_field('var_icon_background'),
              'size' => get_sub_field('var_icon_size'),
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="17"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-17/plx-17' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('18')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'multi_mode' => get_sub_field('step_multi_mode'),
            'has_form_title' => get_sub_field('has_step_form_title'),
            'form_title' => get_sub_field('step_form_title'),
            'form_id' => get_sub_field('step_form_id'),
            'align' => $align,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="18"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-18/plx-18' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('19') && $modules['pricing-plan']['enable_pricing-plan']): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $layout = get_row_layout();
          $type = null;
          if (!isset($counters[$layout])) {
            $counters[$layout] = 1;
          } else {
            $counters[$layout]++;
          }
          $key = 'c-19-instance-' . $counters[$layout];
          $post_type_slug = 'pricing-plan';
          $filtered_posts = get_sub_field('posts');
          $args = [
            'post_type' => $post_type_slug,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'menu_order',
            'order' => 'ASC',
          ];
          if (!empty($filtered_posts)) {
            $post_ids = array_map(function ($post) {
              return $post->ID;
            }, $filtered_posts);
            $args['post__in'] = $post_ids;
            $args['orderby'] = 'post__in'; // keep post order from relationship field
          }
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
            ],
            'key' => $key,
            'query_args' => $args,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="19"
            class="<?= cn('section', $classes) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-19/plx-19' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('20') && $modules['faq']['enable_faq']): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          //Remove Background Image and Classes if variant is Aside
          $is_faq_aside = (
            is_layout_match('20')
            && (string) get_sub_field('var_type') === '1'
          );

        if ($is_faq_aside) {
          $styles = preg_replace(
            '/--background-image:\s*url\([^)]+\);?/',
            '',
            $styles
          );

          $classes = str_replace('has-overlay', '', $classes);
          $classes = str_replace('has-background-image', '', $classes);
        }

        $type = null;
          $accordion_mode = get_sub_field('accordion_mode');
          $post_type_slug = 'faq';
          $filtered_posts = get_sub_field('posts');
          $args = [
            'post_type' => $post_type_slug,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'orderby' => 'menu_order',
            'order' => 'ASC',
          ];
          if (!empty($filtered_posts)) {
            $post_ids = array_map(function ($post) {
              return $post->ID;
            }, $filtered_posts);
            $args['post__in'] = $post_ids;
            $args['orderby'] = 'post__in'; // keep post order from relationship field
          }
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--aside',
            default => null,
          };
        $is_aside = ($type === 'variant--aside');

        $aside_image = null;

        if ($is_aside) {
          $aside_image = get_sub_field('aside_image') ?: null;
        }

          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'accordion_mode' => get_sub_field('accordion_mode'),
            'query_args' => $args,
            'type' => $type,
          ];
        if ($type === 'variant--aside') {
          $template_data['aside_image'] = get_sub_field('aside_image');
        }
          ?>
          <section
            <?= $id ?>
            data-plx="20"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-20/plx-20' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('21')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'steps' => get_sub_field('steps') ?: [],
            'type' => $type,
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="21"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-21/plx-21' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('22') && $modules['calendar']['enable_calendar']): ?>
            <?php
            [$styles, $classes, $id] = generate_block_options();
            $post_type_slug = MODULES['calendar']['slug'];
            $filtered_posts = get_sub_field('posts');
            $today = date('Ymd');
            $args = [
              'post_type' => $post_type_slug,
              'posts_per_page' => -1,
              'post_status' => 'publish',
              'meta_query' => [
                'key' => 'date',
                'value' => $today,
                'type' => 'DATE',
                'compare' => '>=',
              ],
              'meta_key' => 'date',
              'orderby' => 'meta_value_num',
              'order' => 'ASC',
            ];
            if (!empty($filtered_posts)) {
              $post_ids = array_map(function ($post) {
                return $post->ID;
              }, $filtered_posts);
              $args['post__in'] = $post_ids;
              $args['orderby'] = 'post__in'; // keep post order from relationship field
            }
            $type = null;
            $template_data = [
              'content' => [
                'has_caption' => get_sub_field('has_caption'),
                'caption' => get_sub_field('caption'),
                'has_title' => get_sub_field('has_title'),
                'title' => get_sub_field('title'),
                'has_body' => get_sub_field('has_body'),
                'body' => get_sub_field('body'),
                'has_footer' => get_sub_field('has_footer'),
                'button_group' => get_sub_field('button_group') ?: [],
              ],
              'query_args' => $args,
              'type' => $type,
            ];
            ?>
            <section
              <?= $id ?>
              data-plx="22"
              class="<?= cn('section', $classes, $type) ?>"
              <?= $styles ?>
            >
              <?php component('blocks/plx-22/plx-22' . ($type ? '-' . $type : ''), $template_data); ?>
            </section>
        <?php endif; ?>


        <?php if (is_layout_match('23') && $modules['blog']['enable_blog']): ?>
            <?php
            [$styles, $classes] = generate_block_options();
            $post_type_slug = MODULES['blog']['slug'];
            $filtered_posts = get_sub_field('posts');
            $args = [
              'post_type' => $post_type_slug,
              'posts_per_page' => 12,
              'paged' => 1,
              'post_status' => 'publish',
              'orderby' => 'menu_order',
              'order' => 'ASC',
            ];
            if (!empty($filtered_posts)) {
              $post_ids = array_map(function ($post) {
                return $post->ID;
              }, $filtered_posts);
              $args['post__in'] = $post_ids;
              $args['orderby'] = 'post__in'; // keep post order from relationship field
            }
            $terms = get_terms([
              'taxonomy' => 'blog_categories',
              'hide_empty' => true,
            ]);
            $type = match (get_sub_field('var_type') ?? null) {
              '1' => 'variant--blox',
              default => null,
            };
            $template_data = [
              'content' => [
                'has_caption' => get_sub_field('has_caption'),
                'caption' => get_sub_field('caption'),
                'has_title' => get_sub_field('has_title'),
                'title' => get_sub_field('title'),
                'has_body' => get_sub_field('has_body'),
                'body' => get_sub_field('body'),
                'has_footer' => get_sub_field('has_footer'),
                'button_group' => get_sub_field('button_group') ?: [],
              ],
              'query_args' => $args,
              'terms' => $terms,
              'type' => $type,
              'options' => [
                'show_author' => get_theme_option('blog_settings'),
              ],
            ];
            ?>
            <section
              <?= $id ?>
              data-plx="23"
              class="<?= cn('section', $classes, $type) ?>"
              <?= $styles ?>
            >
              <?php component('blocks/plx-23/plx-23' . ($type ? '-' . $type : ''), $template_data); ?>
            </section>
        <?php endif; ?>


        <?php if (is_layout_match('24')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = match (get_sub_field('var_type') ?? null) {
            '1' => 'variant--aside',
            '2' => 'variant--split',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'cards' => get_sub_field('cards') ?: [],
            'type' => $type,
            'custom' => [
              'reduce_motion' => get_sub_field('var_reduce_motion'),
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="24"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-24/plx-24' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('25')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'table' => get_sub_field('table') ?: [],
            'type' => $type,
          ];
          $twig_data = $twig_data ?? [];
          $twig_data[] = get_row(true);
          ?>
          <section
            <?= $id ?>
            data-plx="25"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('26')): ?>
          <?php
          [$styles, $classes, $id] = generate_block_options();
          $type = null;
          $align = match (get_sub_field('var_align') ?? null) {
            '1' => 'align--center',
            '2' => 'align--end',
            default => null,
          };
          $template_data = [
            'content' => [
              'has_caption' => get_sub_field('has_caption'),
              'caption' => get_sub_field('caption'),
              'has_title' => get_sub_field('has_title'),
              'title' => get_sub_field('title'),
              'has_body' => get_sub_field('has_body'),
              'body' => get_sub_field('body'),
              'has_footer' => get_sub_field('has_footer'),
              'button_group' => get_sub_field('button_group') ?: [],
            ],
            'type' => $type,
            'variables' => [
              'align' => $align,
              'has_border' => get_sub_field('var_has_border'),
              'has_shadow' => get_sub_field('var_has_shadow'),
              'background' => get_sub_field('var_background') ?: [],
            ],
          ];
          ?>
          <section
            <?= $id ?>
            data-plx="26"
            class="<?= cn('section', $classes, $type) ?>"
            <?= $styles ?>
          >
            <?php component('blocks/plx-26/plx-26' . ($type ? '-' . $type : ''), $template_data); ?>
          </section>
        <?php endif; ?>


        <?php if (is_layout_match('27') && $modules['team']['enable_team']): ?>
            <?php
            [$styles, $classes, $id] = generate_block_options();
            $post_type_slug = MODULES['team']['slug'];
            $filtered_posts = get_sub_field('posts');
            $args = [
              'post_type' => $post_type_slug,
              'posts_per_page' => -1,
              'post_status' => 'publish',
              'orderby' => 'menu_order',
              'order' => 'ASC',
            ];
            if (!empty($filtered_posts)) {
              $post_ids = array_map(function ($post) {
                return $post->ID;
              }, $filtered_posts);
              $args['post__in'] = $post_ids;
              $args['orderby'] = 'post__in'; // keep post order from relationship field
            }
            $type = match (get_sub_field('var_type') ?? null) {
              '1' => 'variant--aside',
              default => null,
            };
            $align = match (get_sub_field('var_align') ?? null) {
              '1' => 'align--center',
              default => null,
            };
            $template_data = [
              'content' => [
                'has_caption' => get_sub_field('has_caption'),
                'caption' => get_sub_field('caption'),
                'has_title' => get_sub_field('has_title'),
                'title' => get_sub_field('title'),
                'has_body' => get_sub_field('has_body'),
                'body' => get_sub_field('body'),
                'has_footer' => get_sub_field('has_footer'),
                'button_group' => get_sub_field('button_group') ?: [],
              ],
              'query_args' => $args,
              'type' => $type,
              'align' => $align,
            ];
            ?>
            <section
              <?= $id ?>
              data-plx="27"
              class="<?= cn('section', $classes, $type) ?>"
              <?= $styles ?>
            >
              <?php component('blocks/plx-27/plx-27' . ($type ? '-' . $type : ''), $template_data); ?>
            </section>
        <?php endif; ?>


    <?php endwhile; ?>
  <?php endif; ?>
</div>
