<?php

/**
 * Functions and utilities relating to blog posts
 */

function get_author_full_name($author_id = null) {
  if (!$author_id) {
    $author_id = get_post_field('post_author');
  }

  $first_name = get_the_author_meta('first_name', $author_id);
  $last_name = get_the_author_meta('last_name', $author_id);
  $username = get_the_author_meta('user_login', $author_id);

  // if both first and last name exist, use them
  if ($first_name && $last_name) {
    return "$first_name $last_name";
  }
  // if only first name exists
  if ($first_name) {
    return $first_name;
  }
  // if only last name exists
  if ($last_name) {
    return $last_name;
  }
  // fallback to username
  return $username;
}

function get_reading_time($content, $words_per_minute = 225) {
  // clean up the WYSIWYG content
  $clean_content = strip_tags($content);

  // count words
  $word_count = str_word_count($clean_content);

  // calculate reading time
  $minutes = ceil($word_count / $words_per_minute);

  // minimum 1 minute
  return max(1, $minutes) . ' ' . __('min read', 'plexible-theme-modules_blog');
}

function generate_table_of_contents($content) {
  if (!$content) {
    return '';
  }

  // Create DOM document
  $doc = new DOMDocument();

  // Load HTML content and suppress warnings
  @$doc->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));

  // Find all headings (h2-h4)
  $headings = $doc->getElementsByTagName('*');
  $toc = [];

  foreach ($headings as $heading) {
    $tag_name = strtolower($heading->tagName);

    // Only process h2-h4 tags
    if (in_array($tag_name, ['h2', 'h3', 'h4'])) {
      // Create unique ID if none exists
      if (!$heading->getAttribute('id')) {
        $id = sanitize_title($heading->textContent);
        $heading->setAttribute('id', $id);
      } else {
        $id = $heading->getAttribute('id');
      }

      $toc[] = [
        'level' => substr($tag_name, 1), // Get number from h2/h3/h4
        'text' => $heading->textContent,
        'id' => $id,
      ];
    }
  }

  // Generate TOC HTML
  if (empty($toc)) {
    return '';
  }

  $output = '<div class="table-of-contents">';
  $output .= '<ul>';

  foreach ($toc as $heading) {
    $output .= sprintf(
      '<li data-level="%s"><a class="link" href="#%s">%s</a></li>',
      esc_attr($heading['level']),
      esc_attr($heading['id']),
      esc_html($heading['text'])
    );
  }

  $output .= '</ul></div>';

  // Return both TOC and modified content
  return [
    'toc' => $output,
    'content' => $doc->saveHTML(),
  ];
}
