<?php

/**
 * Get embed HTML from supported providers (image, video, and Lottie).
 *
 * @param  array       $data
 * @param  string|null $customClass    Optional extra CSS class to apply on the outer wrapper.
 * @param  array       $vimeoOptions   Overrides for vimeo iframe query args.
 * @param  array       $youtubeOptions Overrides for youtube iframe query args.
 * @return string      HTML string for the embed.
 */

function get_figure(
  array $data,
  ?string $customClass = null,
  array $vimeoOptions = [],
  array $youtubeOptions = [],
): string {
  // Normalize provider
  $provider = strtolower(trim($data['provider'] ?? ''));
  if (empty($provider)) {
    return '';
  }

  // Normalize aliases
  $provider = str_replace('-', '_', $provider);
  $aliases = [
    'local' => 'local_video',
    'image' => 'local_image',
  ];
  if (isset($aliases[$provider])) {
    $provider = $aliases[$provider];
  }

  // Find the src in data
  $fieldNames = ['source_' . $provider, 'src_' . $provider];
  $src = '';
  foreach ($fieldNames as $f) {
    if (!empty($data[$f])) {
      $src = $data[$f];
      break;
    }
  }
  if (empty($src)) {
    return '';
  }

  // Default css class map
  $providerMap = [
    'youtube' => 'yt',
    'vimeo' => 'vimeo',
    'local_video' => 'video',
    'local_image' => 'image',
    'lottie' => 'lottie',
    'google_drive' => 'gdrive',
  ];

  // Default embed options
  $defaultsYT = [
    'autoplay' => 1,
    'mute' => 1,
    'controls' => 0,
    'loop' => 1,
    'playsinline' => 1,
    'rel' => 0,
    'playlist'=>$src
  ];
  $defaultsVM = [
    'autoplay' => 1,
    'loop' => 1,
    'muted' => 1,
    'badge' => 0,
    'controls' => 0,
  ];

  // Merge  overrides
  $ytArgs = array_merge($defaultsYT, $youtubeOptions);
  $vmArgs = array_merge($defaultsVM, $vimeoOptions);

  // Build URLs
  $ytQuery = http_build_query($ytArgs);
  $youtubeEmbed = "https://www.youtube-nocookie.com/embed/{$src}?{$ytQuery}";

  $vmQuery = http_build_query($vmArgs);
  $vimeoEmbed = "https://player.vimeo.com/video/{$src}?{$vmQuery}";
  ?>

  <?php ob_start(); ?>
    <div
      class="<?= cn('figure_inner') ?>"
      data-provider="<?= esc_attr($providerMap[$provider] ?? $provider) ?>"
    >
      <?php switch ($provider): case 'youtube': ?>
        <iframe
          src="<?= esc_url($youtubeEmbed) ?>"
          frameborder="0"
          allow="autoplay; fullscreen"
        ></iframe>
      <?php break;case 'vimeo': ?>
        <iframe
          src="<?= esc_url($vimeoEmbed) ?>"
          frameborder="0"
          allow="autoplay; fullscreen"
        ></iframe>
      <?php break;case 'local_video': ?>
        <video
          width="100%"
          height="100%"
          autoplay
          loop
          muted
          playsinline
        >
          <source src="<?= esc_url($src) ?>" type="video/mp4">
          <?php _e('Your browser does not support the video tag.', 'plexible-theme-terms_errors'); ?>
        </video>
      <?php break;case 'local_image': ?>
        <img
          src="<?= esc_url($src) ?>"
          alt="<?= esc_attr($data['alt'] ?? '') ?>"
          loading="lazy"
          decoding="async"
        />
      <?php break;case 'lottie': ?>
        <lottie-player
          src="<?= esc_url($src) ?>"
          loop
          autoplay
        ></lottie-player>
      <?php break;case 'google_drive': ?>
        <iframe
          src="https://drive.google.com/file/d/<?= esc_attr($src) ?>/preview"
          width="100%"
          height="480"
          allow="autoplay; fullscreen"
          frameborder="0"
        ></iframe>
      <?php break;default:
          echo _e('Unsupported media provider.', 'plexible-theme-terms_errors');
      endswitch; ?>
    </div>
  <?php return ob_get_clean();
}

/**
 * Modern version to be used with twig component
 * @see `/components/ui/figure/`
 */
function get_figure_element(array $data, array $vimeoOptions = [], array $youtubeOptions = []): string {
  // Normalize provider
  $provider = strtolower(trim($data['provider'] ?? ''));
  if (empty($provider)) {
    return '';
  }

  // Normalize aliases
  $provider = str_replace('-', '_', $provider);
  $aliases = [
    'local' => 'local_video',
    'image' => 'local_image',
  ];
  if (isset($aliases[$provider])) {
    $provider = $aliases[$provider];
  }

  // Find the src in data
  $fieldNames = ['source_' . $provider, 'src_' . $provider];
  $src = '';
  foreach ($fieldNames as $f) {
    if (!empty($data[$f])) {
      $src = $data[$f];
      break;
    }
  }
  if (empty($src)) {
    return '';
  }

  // Default css class map
  $providerMap = [
    'youtube' => '-yt',
    'vimeo' => '-vimeo',
    'local_video' => '-vid',
    'local_image' => '-img',
    'google_drive' => '-gdrive',
  ];

  $_defaults_youtube = [
    'autoplay' => 0,
    'mute' => 0,
    'controls' => 0,
    'loop' => 1,
    'playsinline' => 1,
    'playlist' => $src, // needed for loop
    'rel' => 0, // no related
  ];
  $_defaults_vimeo = [
    'autoplay' => 1,
    'loop' => 1,
    'muted' => 1,
    'badge' => 0,
    'controls' => 0,
  ];

  $ytArgs = array_merge_deep($_defaults_youtube, $youtubeOptions);
  $vmArgs = array_merge_deep($_defaults_vimeo, $vimeoOptions);

  // Build URLs
  $ytQuery = http_build_query($ytArgs);
  $youtubeEmbed = "https://www.youtube-nocookie.com/embed/{$src}?{$ytQuery}";

  $vmQuery = http_build_query($vmArgs);
  $vimeoEmbed = "https://player.vimeo.com/video/{$src}?{$vmQuery}";
  ?>

  <?php ob_start(); ?>
    <div class="<?= cn('c-figure_inner', $providerMap[$provider] ?? $provider) ?>">
      <?php switch ($provider): case 'youtube': ?>
        <iframe
          src="<?= esc_url($youtubeEmbed) ?>"
          frameborder="0"
          allow="autoplay; fullscreen"
        ></iframe>
      <?php break;case 'vimeo': ?>
        <iframe
          src="<?= esc_url($vimeoEmbed) ?>"
          frameborder="0"
          allow="autoplay; fullscreen"
        ></iframe>
      <?php break;case 'local_video': ?>
        <video
          width="100%"
          height="100%"
          autoplay
          loop
          muted
          playsinline
        >
          <source src="<?= esc_url($src) ?>" type="video/mp4">
          <?php _e('Your browser does not support the video tag.', 'plexible-theme-terms_errors'); ?>
        </video>
      <?php break;case 'local_image': ?>
        <img
          src="<?= esc_url($src) ?>"
          alt="<?= esc_attr($data['alt'] ?? '') ?>"
          loading="lazy"
          decoding="async"
        />
      <?php break;case 'lottie': ?>
        <lottie-player
          src="<?= esc_url($src) ?>"
          loop
          autoplay
        ></lottie-player>
      <?php break;case 'google_drive': ?>
        <iframe
          src="https://drive.google.com/file/d/<?= esc_attr($src) ?>/preview"
          width="100%"
          height="480"
          allow="autoplay; fullscreen"
          frameborder="0"
        ></iframe>
      <?php break;default:
          echo _e('Unsupported media provider.', 'plexible-theme-terms_errors');
      endswitch; ?>
    </div>
  <?php return ob_get_clean();
}
