<?php

/**
 * Functions and utilities relating to weather
 */

/**
 * Fetching from open-meteo API
 * @see https://open-meteo.com/en/docs
 */

function get_weather_description($code) {
  $codes = [
    0 => 'Clear sky',
    1 => 'Mainly clear',
    2 => 'Partly cloudy',
    3 => 'Overcast',
    45 => 'Fog',
    48 => 'Depositing rime fog',
    51 => 'Light drizzle',
    53 => 'Moderate drizzle',
    55 => 'Dense drizzle',
    61 => 'Slight rain',
    63 => 'Moderate rain',
    65 => 'Heavy rain',
    71 => 'Slight snow fall',
    73 => 'Moderate snow fall',
    75 => 'Heavy snow fall',
    80 => 'Rain showers',
    81 => 'Heavy showers',
    95 => 'Thunderstorm',
  ];

  return $codes[$code] ?? 'Unknown conditions';
}

function get_weather_units() {
  return [
    'temperature' => '°C',
    'wind_speed' => 'km/h',
    'rain' => 'mm/h',
    'rain_chance' => '%',
  ];
}

function get_weather_data($lat, $lon) {
  $transient_key = 'weather_' . md5($lat . '_' . $lon);

  // Allow ?nocache=true to force a fresh API pull
  $skip_cache = isset($_GET['nocache']) && $_GET['nocache'] === 'true';

  if (!$skip_cache) {
    $cached = get_transient($transient_key);
    if ($cached !== false) {
      return $cached;
    }
  }

  $url = "https://api.open-meteo.com/v1/forecast?latitude={$lat}&longitude={$lon}&current=temperature_2m,wind_speed_10m,wind_direction_10m,weather_code,precipitation_probability,rain&timezone=auto&models=best_match";

  $response = wp_remote_get($url);

  if (is_wp_error($response)) {
    return false;
  }

  $body = wp_remote_retrieve_body($response);
  $data = json_decode($body, true);

  if (!isset($data['current'])) {
    return false;
  }

  $weather = [
    'temperature' => round($data['current']['temperature_2m'], 0),
    'wind_speed' => round($data['current']['wind_speed_10m'], 0),
    'wind_direction' => $data['current']['wind_direction_10m'],
    'weather_code' => $data['current']['weather_code'],
    'rain' => $data['current']['rain'],
    'rain_chance' => round($data['current']['precipitation_probability'], 0),
    'description' => get_weather_description($data['current']['weather_code']),
  ];

  // cache for 1 hour
  set_transient($transient_key, $weather, HOUR_IN_SECONDS);

  return $weather;
}
