<?php

/**
 * Normalize a single ACF rule
 */
function acf_normalize_rule(array $rawRule): array {
  if (!array_key_exists('field', $rawRule)) {
    throw new InvalidArgumentException('ACF rule must include a "field" key.');
  }

  $fieldKey = (string) $rawRule['field'];
  $operator = array_key_exists('operator', $rawRule) ? (string) $rawRule['operator'] : '==';
  $value = $rawRule['value'] ?? '';

  return [
    'field' => $fieldKey,
    'operator' => $operator,
    'value' => $value,
  ];
}

/**
 * Shape checks
 */
function acf_is_rule(array $maybeRule): bool {
  return array_key_exists('field', $maybeRule);
}

function acf_is_group(array $maybeGroup): bool {
  // A group is a list of rules.
  return array_is_list($maybeGroup) && isset($maybeGroup[0]) && is_array($maybeGroup[0]) && acf_is_rule($maybeGroup[0]);
}

function acf_is_groups(array $maybeGroups): bool {
  // Groups is a list of groups (each group is a list of rules).
  return array_is_list($maybeGroups) &&
    isset($maybeGroups[0]) &&
    is_array($maybeGroups[0]) &&
    acf_is_group($maybeGroups[0]);
}

/**
 * Build a single AND group for ACF 'conditional_logic'.
 */
function acf_if(array ...$rawConditionsOrSingleGroup): array {
  $rawRules = $rawConditionsOrSingleGroup;

  if (count($rawConditionsOrSingleGroup) === 1 && acf_is_group($rawConditionsOrSingleGroup[0])) {
    $rawRules = $rawConditionsOrSingleGroup[0];
  }

  // Validate every item is a rule
  foreach ($rawRules as $index => $maybeRule) {
    if (!is_array($maybeRule) || !acf_is_rule($maybeRule)) {
      throw new InvalidArgumentException("acf_if(): argument at position {$index} is not a valid rule.");
    }
  }

  $normalizedRules = array_map('_acf_normalize_rule', $rawRules);

  return [$normalizedRules];
}

/**
 * Build OR-of-AND groups for ACF 'conditional_logic'.
 */
function acf_or(array ...$rawGroupsOrIfResults): array {
  $normalizedGroups = [];

  foreach ($rawGroupsOrIfResults as $position => $rawGroupOrGroups) {
    if (acf_is_groups($rawGroupOrGroups)) {
      // Input looks like output of acf_if(): [ [rule, ...] ]
      foreach ($rawGroupOrGroups as $rawGroup) {
        $normalizedGroups[] = array_map('_acf_normalize_rule', $rawGroup);
      }
      continue;
    }

    if (acf_is_group($rawGroupOrGroups)) {
      // Input is a single raw group: [rule, rule, ...]
      $normalizedGroups[] = array_map('_acf_normalize_rule', $rawGroupOrGroups);
      continue;
    }

    throw new InvalidArgumentException(
      "acf_or(): argument at position {$position} must be a group (list of rules) or the result of acf_if(...).",
    );
  }

  return $normalizedGroups;
}
