// Define color palette for maps
const mapColors = {
    dark: {
      geometry: '#212121',
      labelsTextFill: '#757575',
      labelsTextStroke: '#212121',
      water: '#000000',
      roadFill: '#2c2c2c',
      roadTextFill: '#8a8a8a',
      park: '#181818',
      parkText: '#616161',
      parkStroke: '#1b1b1b',
    },
    light: {
      geometry: '#e5e5e5',
      labelsTextFill: '#5f6368',
      labelsTextStroke: '#ffffff',
      water: '#cde6f7',
      roadFill: '#ffffff',
      roadTextFill: '#5f6368',
      park: '#d0f0c0',
      parkText: '#4b5320',
      parkStroke: '#b0c4de',
    },
  };
/**
 * Build a map style array for Google Maps
 * @param {string} variant - 'dark' or 'light'
 */
function getMapStyles(variant = 'default') {
  const colors = mapColors[variant];
  if(variant === 'default') return [];
  return [
    { elementType: 'geometry', stylers: [{ color: colors.geometry }] },
    { elementType: 'labels.icon', stylers: [{ visibility: 'off' }] },
    { elementType: 'labels.text.fill', stylers: [{ color: colors.labelsTextFill }] },
    { elementType: 'labels.text.stroke', stylers: [{ color: colors.labelsTextStroke }] },
    {
      featureType: 'poi.park',
      elementType: 'geometry',
      stylers: [{ color: colors.park }],
    },
    {
      featureType: 'poi.park',
      elementType: 'labels.text.fill',
      stylers: [{ color: colors.parkText }],
    },
    {
      featureType: 'poi.park',
      elementType: 'labels.text.stroke',
      stylers: [{ color: colors.parkStroke }],
    },
    {
      featureType: 'road',
      elementType: 'geometry.fill',
      stylers: [{ color: colors.roadFill }],
    },
    {
      featureType: 'road',
      elementType: 'labels.text.fill',
      stylers: [{ color: colors.roadTextFill }],
    },
    {
      featureType: 'water',
      elementType: 'geometry',
      stylers: [{ color: colors.water }],
    },
    {
      featureType: 'water',
      elementType: 'labels.text.fill',
      stylers: [{ color: colors.labelsTextFill }],
    },
    // Add other feature types as needed
  ];
}
/**
 * Initialize a single Google Map for a given element.
 * @param {HTMLElement} el - The .acf-map element.
 * @param variant
 */

function initMap(el,variant = 'default') {
  // Find all markers within this map
  const markers = el.querySelectorAll('.marker');
  const zoom = el.dataset.zoom || 16;

  // Map styles
  const styles = getMapStyles(variant);
  // Create the map
  const map = new google.maps.Map(el, {
    zoom: parseInt(zoom),
    mapTypeId: google.maps.MapTypeId.ROADMAP,
    styles: styles,
  });

  // Track markers
  map.markers = [];

  // Add each marker
  markers.forEach((marker) => addMarker(marker, map));

  // Center map on all markers
  centerMap(map);
}

/**
 * Add a marker to the map.
 * @param {HTMLElement} markerEl - The .marker element.
 * @param {google.maps.Map} map - The Google Map instance.
 */
function addMarker(markerEl, map) {
  const lat = parseFloat(markerEl.dataset.lat);
  const lng = parseFloat(markerEl.dataset.lng);
  const position = { lat, lng };

  // Create marker
  const marker = new google.maps.Marker({
    position,
    map,
  });

  // Add to markers array
  map.markers.push(marker);

  // If marker has HTML, add info window
  if (markerEl.innerHTML) {
    const infowindow = new google.maps.InfoWindow({
      content: markerEl.innerHTML,
    });

    marker.addListener('click', () => {
      infowindow.open(map, marker);
    });
  }
}

/**
 * Center the map on all markers.
 * @param {google.maps.Map} map - The Google Map instance.
 */
function centerMap(map) {
  const bounds = new google.maps.LatLngBounds();
  map.markers.forEach((marker) => {
    bounds.extend(marker.getPosition());
  });

  if (map.markers.length === 1) {
    map.setCenter(bounds.getCenter());
  } else {
    map.fitBounds(bounds);
  }
}
function resolveMapVariant(el) {
  return el.dataset?.theme === 'dark' || el.dataset?.theme === 'light'
    ? el.dataset.theme
    : 'default';
}
/**
 * Initialize all ACF Google Maps on the page.
 * This function is called automatically by the Google Maps API after it loads.
 */
export function initAllMaps() {
  // Example: toggle maps based on user choice or theme
  document.querySelectorAll('.acf-map').forEach(el => {
    const variant = resolveMapVariant(el); // e.g., 'light' or 'dark'
    initMap(el, variant);
  });

}

export function init() {
  if (window.google?.maps) {
    initAllMaps();
  }
}
