/**
 * Classic Blocks
 *
 * Javascript related to classic (legacy) blocks
 */

import { lenis } from '../motion/smoothscroll';

// header
function handleHeader() {
  const header = document.querySelector('[data-js-controller="header"]');

  if (!header) return;

  const mobileMenuToggle = header.querySelector('.mobile-menu-toggle');

  if (!mobileMenuToggle) return;

  const toggleMenu = () => {
    const isExpanded = mobileMenuToggle.getAttribute('data-expanded') === 'true';
    const newState = !isExpanded;

    mobileMenuToggle.setAttribute('data-expanded', newState.toString());
    mobileMenuToggle.setAttribute('aria-label', newState ? 'Close menu' : 'Open menu');
    header.classList.toggle('menu-visible');

    // disable lenis scroll when menu active
    newState ? lenis.stop() : lenis.start();
  };

  mobileMenuToggle.addEventListener('click', toggleMenu);

  const handleScroll = () => {
    const scrolled = window.scrollY > 50;
    header.setAttribute('data-scrolled', scrolled ? 'true' : 'false');
  };

  window.addEventListener('scroll', handleScroll);

}

// modal
function initializeModal() {
  const modalElement = document.querySelector('[data-js-controller="modal"]');
  if (!modalElement) return;

  const modalBackground = modalElement.querySelector('[data-js-target="modal.background"]');
  const modalContainer = modalElement.querySelector('[data-js-target="modal.container"]');

  function toggleModal() {
    const isOpen = modalElement.dataset.isOpen === 'true';
    modalElement.dataset.isOpen = (!isOpen).toString();

    if (!isOpen) {
      document.body.style.overflow = 'hidden';
      lenis.stop();
    } else {
      document.body.style.overflow = '';
      lenis.start();
    }
  }

  modalBackground.addEventListener('click', toggleModal);
  modalContainer.addEventListener('click', (event) => {
    event.stopPropagation();
  });

  return {
    element: modalElement,
    toggle: toggleModal,
  };
}

// lightbox gallery
function initializeLightboxGallery() {
  const modal = initializeModal();
  if (!modal) return;

  const gallery = document.querySelector('[data-js-controller="gallery"]');
  const galleryItems = gallery.querySelectorAll('[data-js-target="gallery.item"]');

  function openLightbox(index) {
    modal.toggle();

    if (typeof galleryLightbox !== 'undefined') {
      galleryLightbox.slideTo(index, 0);
    }
  }

  galleryItems.forEach((item, index) => {
    item.addEventListener('click', (event) => {
      const clickedIndex = parseInt(event.currentTarget.dataset.index, 10);
      openLightbox(clickedIndex);
    });
  });
}

// input switch
const controlInputSwitch = (element, callback) => {
  const inputs = element.querySelectorAll('input[type="radio"]');
  const items = element.querySelectorAll('.label-switch-item');

  const updateActiveState = (selectedInput) => {
    items.forEach((item) => {
      item.classList.remove('active');
    });

    const selectedItem = selectedInput.closest('.label-switch-item');
    if (selectedItem) {
      selectedItem.classList.add('active');
    }

    const value = selectedInput.value;
    if (callback) callback(value);
  };

  inputs.forEach((input) => {
    input.addEventListener('change', (e) => {
      updateActiveState(e.target);
    });
  });

  // set initial active state
  const initialCheckedInput = element.querySelector('input[type="radio"]:checked');
  if (initialCheckedInput) {
    updateActiveState(initialCheckedInput);
  }
};

// concierge
function handleConcierge() {
  const concierge = document.querySelector('[data-js-controller="concierge"]');
  if (!concierge) return;

  const controls = concierge.querySelector('[data-js-target="concierge.controls"]');
  const stepsContainer = concierge.querySelector('.steps');
  const steps = concierge.querySelectorAll('.step');
  const formContainer = concierge.querySelector('.form');
  const formElement = formContainer?.querySelector('form');
  const prevButton = controls?.querySelector('.--prev');
  const nextButton = controls?.querySelector('.--next');
  const submitButton = controls?.querySelector('.--submit');

  // Check if multiple choices are allowed
  const allowMultipleChoices = concierge.hasAttribute('data-js-allow-multiple-choices');

  let currentStepIndex = 0;
  const selectedChoices = {};

  const initConciergeForm = () => {
    if (!controls || !prevButton || !nextButton || !submitButton || !formContainer) {
      console.error('Missing essential concierge form elements');
      return;
    }

    // Prevent default form submission
    const form = formContainer.querySelector('form');
    if (form) {
      form.addEventListener('submit', (e) => {
        e.preventDefault();
      });
    }

    // Hide all steps first except the first one
    steps.forEach((step, index) => {
      step.setAttribute('data-active', 'false');
    });
    steps[0].setAttribute('data-active', 'true');

    // Set initial states using data-active
    formContainer.setAttribute('data-active', 'false');
    submitButton.setAttribute('data-active', 'false');
    nextButton.setAttribute('data-active', 'true');

    // Disable previous button on initialization
    prevButton.disabled = true;

    // Event listeners
    prevButton.addEventListener('click', goToPreviousStep);
    nextButton.addEventListener('click', goToNextStep);
    submitButton.addEventListener('click', submitForm);

    // Choice selection listeners with single/multi-select handling
    steps.forEach((step, stepIndex) => {
      const choices = step.querySelectorAll('.choice input[type="checkbox"]');
      choices.forEach((choice) => {
        choice.addEventListener('change', (e) => {
          if (!allowMultipleChoices && e.target.checked) {
            // If single-select mode, uncheck all other choices in this step
            choices.forEach((otherChoice) => {
              if (otherChoice !== e.target) {
                otherChoice.checked = false;
              }
            });
          }
          updateNavigationState(stepIndex);
          captureStepSelections(stepIndex);
        });
      });

      // If single-select mode, convert checkboxes to radio-like behavior
      if (!allowMultipleChoices) {
        choices.forEach((choice) => {
          choice.setAttribute('data-selection-mode', 'single');
        });
      }
    });

    updateNavigationState(0);
  };

  const captureStepSelections = (stepIndex) => {
    const step = steps[stepIndex];
    // const stepTitle = step.querySelector('.copy--title')?.textContent.trim() || '';
    const choices = step.querySelectorAll('.choice input[type="checkbox"]:checked');

    const stepSelections = Array.from(choices).map((choice) => {
      const choiceContainer = choice.closest('.choice');
      const title = choiceContainer?.querySelector('.copy--title')?.textContent.trim() || '';
      const description = choiceContainer?.querySelector('.wysiwyg')?.textContent.trim() || '';
      return `${title}: ${description}`;
    });

    selectedChoices[`step_${stepIndex + 1}`] = stepSelections;

    if (currentStepIndex === steps.length - 1) {
      populateFormidableFields();
      populateFormPreview();
    }
  };

  const populateFormidableFields = () => {
    const hiddenField = document.getElementById('field_concierge_selections');
    if (!hiddenField) return;

    const content = Object.entries(selectedChoices)
      .map(([stepKey, selections]) => {
        const stepNumber = stepKey.split('_')[1];
        const selectionsList = selections
          .map((choice) => {
            const choiceTitle = choice.split(':')[0];
            return `<li>${choiceTitle}</li>`;
          })
          .join('');

        return `
          <h3>Step ${stepNumber}</h3>
          <ul>${selectionsList}</ul>
        `;
      })
      .join('');

    hiddenField.value = content;
  };

  const populateFormPreview = () => {
    const preview = concierge?.querySelector('.preview--content');
    if (!preview) return;

    const content = Object.entries(selectedChoices)
      .map(([stepKey, selections]) => {
        const stepNumber = stepKey.split('_')[1];
        const selectionsList = selections
          .map((choice) => {
            const choiceTitle = choice.split(':')[0];
            return `<li class="list--item">${choiceTitle}</li>`;
          })
          .join('');

        return `
          <div class="list variant--disc">
            <h3 class="list--title">Step ${stepNumber}</h3>
            <ul class="list--list">${selectionsList}</ul>
          </div>
        `;
      })
      .join('');

    preview.innerHTML = content;
  };

  const updateNavigationState = (stepIndex) => {
    const hasSelectedChoice = validateStepChoices(stepIndex);

    // Just handle button disabled states here
    nextButton.disabled = !hasSelectedChoice;
    prevButton.disabled = stepIndex === 0;

    // Keep next button visible during all steps
    nextButton.setAttribute('data-active', 'true');
    // Keep submit button hidden during steps
    submitButton.setAttribute('data-active', 'false');
  };

  const validateStepChoices = (stepIndex) => {
    const step = steps[stepIndex];
    const choices = step.querySelectorAll('.choice input[type="checkbox"]');
    return Array.from(choices).some((choice) => choice.checked);
  };

  const goToPreviousStep = () => {
    if (formContainer.getAttribute('data-active') === 'true') {
      // Return to last step from form view
      formContainer.setAttribute('data-active', 'false');
      submitButton.setAttribute('data-active', 'false');
      nextButton.setAttribute('data-active', 'true');
      stepsContainer.setAttribute('data-active', 'true');

      currentStepIndex = steps.length - 1;
      steps[currentStepIndex].setAttribute('data-active', 'true');
    } else if (currentStepIndex > 0) {
      // Navigate between steps
      steps[currentStepIndex].setAttribute('data-active', 'false');
      currentStepIndex--;
      steps[currentStepIndex].setAttribute('data-active', 'true');
    }

    updateNavigationState(currentStepIndex);
  };

  const goToNextStep = () => {
    if (!validateStepChoices(currentStepIndex)) return;

    // If we're not on the last step, go to next step
    if (currentStepIndex < steps.length - 1) {
      steps[currentStepIndex].setAttribute('data-active', 'false');
      currentStepIndex++;
      steps[currentStepIndex].setAttribute('data-active', 'true');
      updateNavigationState(currentStepIndex);
    }
    // If we're on the last step and it's validated, show the form
    else if (currentStepIndex === steps.length - 1) {
      // Hide the last step
      steps[currentStepIndex].setAttribute('data-active', 'false');
      // Show the form and submit button, hide next button
      nextButton.setAttribute('data-active', 'false');
      stepsContainer.setAttribute('data-active', 'false');
      formContainer.setAttribute('data-active', 'true');
      submitButton.setAttribute('data-active', 'true');

      // Capture final selections and update form
      captureStepSelections(currentStepIndex);
    }
  };

  const submitForm = (e) => {
    // Prevent any default button behavior
    e.preventDefault();

    const formSubmitButton = formContainer.querySelector('.frm_button_submit');
    if (formSubmitButton) {
      // Explicitly trigger the click event
      formSubmitButton.dispatchEvent(
        new MouseEvent('click', {
          bubbles: true,
          cancelable: true,
          view: window,
        })
      );
    } else {
      console.error('Form submit button not found');
    }
  };

  initConciergeForm();
}

// pricing table
const handlePricingTable = () => {
  const pricingTables = document.querySelectorAll('[data-js-controller="pricingTable"]');

  pricingTables.forEach((table) => {
    const tableSwitch = table.querySelector('[data-js-target="pricingTable.switch"]');

    // exit if no table switch
    if (!tableSwitch) return;

    const container = table.querySelector('[data-js-target="pricingTable.columns"]');
    const yearlyPrices = container.querySelectorAll('[data-js-target="pricingTable.price.year"]');
    const monthlyPrices = container.querySelectorAll('[data-js-target="pricingTable.price.month"]');

    const updatePriceVisibility = (isMonthly) => {
      const showPrices = isMonthly ? monthlyPrices : yearlyPrices;
      const hidePrices = isMonthly ? yearlyPrices : monthlyPrices;

      showPrices.forEach((el) => {
        el.setAttribute('aria-hidden', 'false');
        el.dataset.visible = 'true';
      });

      hidePrices.forEach((el) => {
        el.setAttribute('aria-hidden', 'true');
        el.dataset.visible = 'false';
      });
    };

    controlInputSwitch(tableSwitch, (value) => {
      updatePriceVisibility(value === 'month');
    });
  });
};

// FAQ
const handleFAQ = () => {
  const accordion = document.querySelector('[data-js-controller="accordion"]');

  if (!accordion) return;

  const accordionItems = [...accordion.querySelectorAll('[data-js-controller="accordionItem"]')];
  const allowMultipleExpanded = accordion.hasAttribute('data-js-allow-multiple-expanded');

  const toggleAccordion = (item, toggle, content) => {
    const isExpanded = item.getAttribute('aria-expanded') === 'true';

    if (!allowMultipleExpanded && !isExpanded) {
      accordionItems
        .filter(
          (otherItem) => otherItem !== item && otherItem.getAttribute('aria-expanded') === 'true'
        )
        .forEach((otherItem) => {
          const otherToggle = otherItem.querySelector('[data-js-target="accordionItem.toggle"]');
          const otherContent = otherItem.querySelector('[data-js-target="accordionItem.content"]');
          collapseItem(otherItem, otherToggle, otherContent);
        });
    }

    isExpanded ? collapseItem(item, toggle, content) : expandItem(item, toggle, content);
  };

  const expandItem = (item, toggle, content) => {
    [item, toggle].forEach((el) => el.setAttribute('aria-expanded', 'true'));
    content.style.height = `${content.scrollHeight}px`;
    content.addEventListener('transitionend', () => (content.style.height = 'auto'), {
      once: true,
    });
  };

  const collapseItem = (item, toggle, content) => {
    [item, toggle].forEach((el) => el.setAttribute('aria-expanded', 'false'));
    content.style.height = `${content.scrollHeight}px`;
    requestAnimationFrame(() => (content.style.height = '0'));
  };

  accordionItems.forEach((item) => {
    const toggle = item.querySelector('[data-js-target="accordionItem.toggle"]');
    const content = item.querySelector('[data-js-target="accordionItem.content"]');

    item.addEventListener('click', (event) => {
      if (event.target === toggle || toggle.contains(event.target)) {
        toggleAccordion(item, toggle, content);
      }
    });
  });
};

// grid overlay
const handleGridOverlay = () => {
  const ctrl = document.querySelector('[data-js-controller="grid-overlay"]');
  if (!ctrl) return;
  const res = ctrl.querySelector('[data-js-target="res"]');
  if (!res) return;

  function update() {
    res.style.visibility = 'visible';
    res.textContent = `${window.innerWidth}x${window.innerHeight}`;
  }

  let ticking = false;
  window.addEventListener(
    'resize',
    () => {
      if (!ticking) {
        window.requestAnimationFrame(() => {
          update();
          ticking = false;
        });
        ticking = true;
      }
    },
    { passive: true }
  );

  update();
};

/**
 * Init functions
 */
document.addEventListener('DOMContentLoaded', () => {
  handleHeader();
  handlePricingTable();
  initializeLightboxGallery();
  handleConcierge();
  // handleFAQ();
  handleGridOverlay();
});
