import { scroll, transform } from 'https://cdn.jsdelivr.net/npm/motion@12.5.0/+esm';

import gsap from 'gsap';

// gsap plugins
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { SplitText } from 'gsap/SplitText';
import { Observer } from 'gsap/Observer';

/**
 * PLX 24
 */
function animateCardsParallax() {
  const cardsParallaxRef = document.querySelector('[data-plx="24"]');

  if (!cardsParallaxRef) return;

  const cardsParallaxContainer = cardsParallaxRef.querySelectorAll('.card-container');
  const cardsParallaxCard = cardsParallaxRef.querySelectorAll('.card-container > .card');

  // set card container height + largest card height, update on window resize
  const updateHeights = () => {
    cardsParallaxContainer.forEach((container) =>
      container.style.setProperty('--height', `${container.offsetHeight}px`)
    );

    const maxHeight = Math.max(
      ...Array.from(cardsParallaxContainer).map((container) => container.offsetHeight)
    );
    cardsParallaxRef.style.setProperty('--max-height', `${maxHeight}px`);
  };

  updateHeights();
  window.addEventListener('resize', updateHeights);

  // scale cards based on section scroll progression
  scroll(
    (progress) => {
      cardsParallaxRef.style.setProperty('--scroll-progress', progress);

      cardsParallaxCard.forEach((card, index) => {
        const targetScale = 1 - (cardsParallaxCard.length - index) * 0.05;
        const range = [index * 0.25, 1];
        const cardScale = transform(progress, range, [1, targetScale]);
        card.style.setProperty('--scale', cardScale);
      });
    },
    {
      target: cardsParallaxRef,
      offset: ['start start', 'end end'],
    }
  );

  // scale images based on individual card scroll progression
  cardsParallaxContainer.forEach((container) => {
    scroll(
      (progress) => {
        const image = container.querySelector('.figure');
        const imageScale = transform(progress, [0, 1], [2, 1]);
        image.style.setProperty('--scale', imageScale);
      },
      {
        target: container,
        offset: ['start end', 'start start'],
      }
    );
  });
}

/**
 * PLX 01 - Parallax Grid
 */
function animateBlockOneParallaxGrid() {
  const ref = document.querySelector('[data-js-target="hero.varParallaxGrid"]');

  if (!ref) return;

  const items = gsap.utils.toArray(ref.querySelectorAll('img'));

  if (items.length === 0) {
    console.warn('No images found in parallax grid.');
    return;
  }

  const col1 = items.filter((_, i) => i % 3 === 0); // 1,4,7
  const col2 = items.filter((_, i) => i % 3 === 1); // 2,5,8
  const col3 = items.filter((_, i) => i % 3 === 2); // 3,6,9

  // create a GSAP timeline that will be controlled by `ScrollTrigger`
  const tl = gsap.timeline({
    scrollTrigger: {
      trigger: ref,
      start: 'top bottom',
      end: 'bottom top',
      scrub: 1,
      // markers: true, // for debugging ScrollTrigger start/end points
    },
  });

  const tween = 'linear';

  if (col1.length > 0 || col3.length > 0) {
    tl.fromTo(
      col1.concat(col3),
      { y: '-14rem' }, // starting state
      { y: '0rem', ease: tween, immediateRender: false }, // ending state
      0
    );
  }
  if (col2.length > 0) {
    tl.fromTo(
      col2,
      { y: '0rem' }, // starting state
      { y: '-10rem', ease: tween, immediateRender: false }, // ending state
      0
    );
  }
}

/**
 * PLX 03 - Track Slider
 */
function animateBlockThree() {
  const tracks = document.querySelectorAll('.sticky-element');

  tracks.forEach((track, i) => {
    let trackWrapper = track.querySelectorAll('.track');

    let trackWrapperWidth = () => {
      let width = 0;
      trackWrapper.forEach((el) => (width += el.offsetWidth));
      return width;
    };

    let scrollTween = gsap.to(trackWrapper, {
      x: () => -trackWrapperWidth() + window.innerWidth,
      ease: 'none',
      scrollTrigger: {
        trigger: track,
        //invalidateOnRefresh: true,
        pin: true,
        scrub: 1,
        start: 'center center',
        end: () => '+=' + (track.scrollWidth - window.innerWidth),
        invalidateOnRefresh: true,
        id: 'id-one',
      },
    });

    Observer.create({
      target: trackWrapper,
      type: 'wheel,touch,pointer',
      onPress: (self) => {
        self.startScroll = scrollTween.scrollTrigger.scroll();
        scrollTo = gsap.quickTo(scrollTween.scrollTrigger, 'scroll', {
          duration: 0.5,
          ease: 'power3',
        });
      },
      onDrag: (self) => {
        scrollTo(self.startScroll + (self.startX - self.x) * dragRatio);
      },
    });
  });
}

/**
 * Global
 *
 * Split Lines viewport animation `data-motion="splitLine"`
 */
function animateSplitLinesOnScroll() {
  const targets = gsap.utils.toArray('[data-motion="splitLine"]');

  if (!targets.length) return;

  targets.forEach((element) => {
    // Ensure element is visible if CSS initially hides it.
    // GSAP's autoAlpha will handle visibility during animation.
    gsap.set(element, { visibility: 'visible' });

    if (!element.textContent || element.textContent.trim() === '') return;

    const triggerElement = element.closest('section[data-plx]');

    if (!triggerElement) {
      console.warn(
        `No parent section[data-plx] found for splitLine element:`,
        element,
        `Animation will not be triggered for this element.`
      );
      return;
    }

    let splitInstance; // store the SplitText instance for this textElement

    const enterAnimation = () => {
      // Split the text when the element enters the viewport
      // Revert previous split if it exists (important for re-triggering)
      if (splitInstance && typeof splitInstance.revert === 'function') {
        splitInstance.revert();
      }
      try {
        splitInstance = new SplitText(element, {
          type: 'lines',
          mask: 'lines',
        });
        if (splitInstance.lines && splitInstance.lines.length > 0) {
          gsap.from(splitInstance.lines, {
            y: '100%',
            // autoAlpha: 0,
            duration: 0.8,
            ease: 'power2.out',
            stagger: 0.15,
          });
        } else {
          // Handle case where SplitText runs but finds no words
          if (splitInstance && typeof splitInstance.revert === 'function') splitInstance.revert();
        }
      } catch (e) {
        console.error('Failed to initialize or animate SplitText for element:', element, e);
        if (splitInstance && typeof splitInstance.revert === 'function') splitInstance.revert();
      }
    };

    const exitAnimation = () => {
      // Revert the split when scrolling up past the element
      if (splitInstance && typeof splitInstance.revert === 'function') {
        splitInstance.revert();
      }
    };

    // create a ScrollTrigger for each element
    ScrollTrigger.create({
      trigger: triggerElement,
      start: 'top 85%',
      end: 'bottom 15%',
      // markers: true, // For debugging
      onEnter: enterAnimation,
      onLeave: exitAnimation,
      onEnterBack: enterAnimation,
      onLeaveBack: exitAnimation,
    });
  });
}

/**
 * Trigger animations on `DOMContentLoaded`
 */
document.addEventListener('DOMContentLoaded', () => {
  animateCardsParallax();
  animateBlockOneParallaxGrid();
  animateSplitLinesOnScroll();
  animateBlockThree();
});
