import { initCube3FilterSliderCarousel } from './swiper.js';

/**
 * Click handler for term filters
 */
async function handleTermFilterClick(event) {
  event.preventDefault();
  const btn = event.currentTarget;

  const {
    ajaxActionName,
    ajaxContainer,
    ajaxUrl,
    ajaxCallback = '',
    ajaxQueryArgs = '{}',
    ajaxNonce,
  } = btn.dataset;

  const isAlreadyActive = btn.dataset.active === 'true';
  const termId = isAlreadyActive ? 'all' : btn.dataset.filterId;

  const layoutRow = btn.closest('.layout-row');
  if (layoutRow) {
    const loadMoreBtn = layoutRow.querySelector('[data-js-target="pagination.loadMore"]');
    if (loadMoreBtn) loadMoreBtn.dataset.ajaxTermId = termId;
  }

  const container = document.querySelector(`.${ajaxContainer}`);
  if (!container) return console.error('Missing content container for filtered posts.');

  // Update active state for all buttons
  btn.closest('.c3-filter-list')
    .querySelectorAll('.filter')
    .forEach((b) => (b.dataset.active = 'false'));

  btn.dataset.active = isAlreadyActive ? 'false' : 'true';
  if (isAlreadyActive) {
    const allBtn = btn.closest('.c3-filter-list').querySelector('[data-filter-id="all"]');
    if (allBtn) allBtn.dataset.active = 'true';
  }

  try {
    container.style.opacity = '0.5';

    const payload = new URLSearchParams({
      action: ajaxActionName,
      security: ajaxNonce,
      term_id: termId,
      loader_callback: ajaxCallback,
      query_args: JSON.stringify(JSON.parse(ajaxQueryArgs)),
    });

    const res = await fetch(ajaxUrl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: payload.toString(),
    });

    const text = await res.text();
    let data;
    try {
      data = JSON.parse(text);
    } catch {
      console.error('Invalid JSON returned:', text);
      throw new Error('Invalid JSON from server');
    }

    if (!data.success) throw new Error(data.data || 'Unknown AJAX error');

    // Replace carousel content
    container.innerHTML = data.data.html || '<p>No posts found.</p>';

    // Re-initialize both sliders inside this container
    initCube3FilterSliderCarousel(`.${ajaxContainer}`);

    container.style.opacity = '1';

    // Reset Load More button
    if (layoutRow) {
      const loadMoreBtn = layoutRow.querySelector('[data-js-target="pagination.loadMore"]');
      if (loadMoreBtn) {
        loadMoreBtn.dataset.page = '1';
        loadMoreBtn.dataset.ajaxTermId = termId;

        if (data.data.max_pages < 2) {
          loadMoreBtn.disabled = true;
          loadMoreBtn.style.display = 'none';
        } else {
          loadMoreBtn.disabled = false;
          loadMoreBtn.style.display = '';
        }
      }
    }
  } catch (err) {
    console.error('Term filter failed:', err);
    container.innerHTML = '<p class="error-message">Error loading posts. Try again.</p>';
    container.style.opacity = '1';
  }
}

/**
 * Initialize term filter buttons
 */
export function initTermFilterButtons() {
  document.querySelectorAll('[data-js-target="filter.termFilter"]').forEach((btn) => {
    btn.addEventListener('click', handleTermFilterClick);
  });
}

// Auto-init on DOMContentLoaded
document.addEventListener('DOMContentLoaded', () => {
  initTermFilterButtons();

  // Only initialize sliders if container exists
  const containers = document.querySelectorAll('[data-js-target="dynamicTwo.slides.carousel"]');
  if (containers.length) {
    containers.forEach((container) => {
      initCube3FilterSliderCarousel(`#${container.id}`);
    });
  }
});
