/**
 * Click handler
 * @param {MouseEvent} event
 */
async function handleLoadMoreClick(event) {
  event.preventDefault();
  const btn = event.currentTarget;
  await loadMorePosts(btn);
}

/**
 * Perform one “load more” cycle for the given button
 * @param {HTMLElement} btn
 */
async function loadMorePosts(btn) {
  // pull from data-attrs
  let {
    ajaxQueryArgs = '{}',
    ajaxComponentArgs = '{}',
    page = '1',
    perPage = '12',
    maxPages = '0',
    ajaxContainer,
    ajaxUrl,
    ajaxNonce,
    ajaxCallback = '',
    ajaxTermId = '',
  } = btn.dataset;

  page = Number(page);
  perPage = Number(perPage);
  maxPages = Number(maxPages);

  // bail if no more pages
  if (maxPages && page >= maxPages) {
    btn.style.display = 'none';
    return;
  }

  const nextPage = page + 1;
  const payload = new URLSearchParams({
    action: 'load_more_posts',
    security: ajaxNonce,
    page: nextPage,
    per_page: perPage,
    loader_callback: ajaxCallback,
    query_args: ajaxQueryArgs,
    component_args: ajaxComponentArgs,
    term_id: ajaxTermId
  });

  const originalText = btn.textContent;
  btn.disabled = true;
  btn.textContent = 'Loading…';

  try {
    const res = await fetch(ajaxUrl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: payload.toString(),
    });

    // if server error, grab text and throw
    if (!res.ok) {
      const text = await res.text();
      console.error('Server error response (500?):', text);
      throw new Error(`HTTP ${res.status}: ${res.statusText}`);
    }

    // try to parse JSON
    let data;
    const txt = await res.text();
    try {
      data = JSON.parse(txt);
      btn.textContent = originalText;
    } catch (parseErr) {
      console.error('Invalid JSON returned by AJAX:', txt);
      throw parseErr;
    }

    if (!data.success) {
      console.error('AJAX returned success=false:', data);
      throw new Error(data.data || 'unknown error');
    }

    // append or remove if empty
    const html = (data.data.html || '').trim();
    if (!html) {
      btn.style.display = 'none';
      return;
    }
    document
      .getElementById(ajaxContainer)
      .insertAdjacentHTML('beforeend', html);

    // update counters
    btn.dataset.page = String(nextPage);
    btn.dataset.maxPages = String(data.data.max_pages || maxPages);

    if (nextPage >= Number(btn.dataset.maxPages)) {
      btn.style.display = 'none';
    } else {
      btn.disabled = false;
    }
  } catch (err) {
    console.error('loadMorePosts error:', err);
    btn.textContent = 'Error, try again';
    btn.disabled = false;
  }
}

/**
 * Init
 */
function initLoadMoreButtons() {
  const buttons = document.querySelectorAll(
    '[data-js-target="pagination.loadMore"]'
  );
  buttons.forEach((btn) => btn.addEventListener('click', handleLoadMoreClick));
}
document.addEventListener('DOMContentLoaded', initLoadMoreButtons);
