import { gsap } from 'gsap';
import { SplitText } from 'gsap/SplitText';

export class BlurScrollEffect {
  constructor(textElement) {
    if (!textElement || !(textElement instanceof HTMLElement)) {
      throw new Error('Invalid text element provided to BlurScrollEffect.');
    }

    this.textElement = textElement;
    this.initializeEffect();
  }

  initializeEffect() {
    // Whenever window resizes, kill old triggers and recreate the effect
    window.addEventListener('resize', () => {
      this.kill();
      this.initializeEffect();
    });

    // Split the text into words and chars using GSAP SplitText
    this.splitter = new SplitText(this.textElement, {
      type: 'words,chars',
    });

    // Run the animation
    this.scroll();
  }

  scroll() {
    const chars = this.splitter.chars;

    // Clear old animation and ScrollTrigger if any
    if (this.tween) {
      this.tween.scrollTrigger.kill();
      this.tween.kill();
    }

    this.tween = gsap.fromTo(
      chars,
      {
        filter: 'blur(10px) brightness(0%)',
        willChange: 'filter, opacity',
        opacity: 0,
      },
      {
        filter: 'blur(0px) brightness(100%)',
        opacity: 1,
        stagger: 0.05,
        ease: 'none',
        scrollTrigger: {
          trigger: this.textElement,
          start: 'top bottom-=15%',
          end: 'bottom center+=15%',
          scrub: true,
        },
      }
    );
  }

  kill() {
    if (this.tween) {
      this.tween.scrollTrigger.kill();
      this.tween.kill();
    }
    if (this.splitter) {
      this.splitter.revert(); // Revert the split text back to original text
    }
  }
}
