import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { lenis } from './smoothscroll';

function scrollProgress() {
  const vh = window.innerHeight;

  const update = () => {
    document.querySelectorAll('[data-scroll-progress]').forEach((el) => {
      const rect = el.getBoundingClientRect();
      const bottom = rect.bottom;

      let progress = (vh - bottom) / vh;
      progress = Math.min(Math.max(progress, 0), 1);

      el.style.setProperty('--scroll-progress', progress);
    });
  };

  lenis.on('scroll', update);
  window.addEventListener('resize', update);
}

function isElementInView({
  selector = '[data-scroll]',
  className = 'is-inview',
  defaultOffset = '0,0', // "topOffset,bottomOffset" in px or %, e.g., "0,100%"
} = {}) {
  if (!gsap || !ScrollTrigger) {
    console.error('[isElementInView] GSAP + ScrollTrigger required.');
    return;
  }

  const parseOffsets = (str) => {
    const parts = String(str || defaultOffset)
      .split(/[\s,]+/)
      .filter(Boolean);
    const [a = '0', b = '0'] = parts.length === 1 ? [parts[0], parts[0]] : parts;
    const norm = (v) => {
      const m = String(v)
        .trim()
        .match(/^(-?\d+(?:\.\d+)?)(%|px)?$/i);
      return !m
        ? { val: 0, unit: 'px' }
        : { val: parseFloat(m[1]), unit: (m[2] || 'px').toLowerCase() };
    };
    return { top: norm(a), bottom: norm(b) };
  };

  const startStr = ({ top }) => {
    const s = `${Math.abs(top.val)}${top.unit}`;
    if (top.val === 0) return 'top bottom';
    return top.val > 0 ? `top bottom-=${s}` : `top bottom+=${s}`;
  };

  const endStr = ({ bottom }) => {
    const s = `${Math.abs(bottom.val)}${bottom.unit}`;
    if (bottom.val === 0) return 'bottom top';
    return bottom.val > 0 ? `bottom top+=${s}` : `bottom top-=${s}`;
  };

  document.querySelectorAll(selector).forEach((el) => {
    const offsets = parseOffsets(el.getAttribute('data-scroll-offset'));
    const repeat = el.hasAttribute('data-scroll-repeat');

    const config = {
      trigger: el,
      start: startStr(offsets),
      end: endStr(offsets),
    };

    if (repeat) {
      config.toggleClass = { targets: el, className };
    } else {
      config.onToggle = (self) => {
        if (self.isActive) {
          el.classList.add(className);
        }
      };
    }

    ScrollTrigger.create(config);
  });

  ScrollTrigger.refresh();
}

function initializeScrollUtils() {
  scrollProgress();
  isElementInView();
}

export const init = initializeScrollUtils;
