/**
 * Registers a delegate event handler on the specified element or document.
 *
 * @param {HTMLElement | Document} $element - The element or document to attach the event handler to.
 * @param {string} eventType - The type of event to listen for (e.g., 'click', 'keydown', etc.).
 * @param {string} selector - The CSS selector for the target elements within the $element.
 * @param {Function} handler - The event handler function to be executed when the event occurs.
 * @param {Boolean} [capturingPhase=false] - See: https://stackoverflow.com/questions/7398290/unable-to-understand-usecapture-parameter-in-addeventlistener
 * @returns {Function} - A function to remove the delegate event handler.
 */
const delegateEvent = ($element, eventType, selector, handler, capturingPhase = false) => {
  const eventHandler = (event) => {
    // Normalize target (some events can originate from text nodes)
    const target =
      event.target && event.target.nodeType === 1
        ? event.target
        : (event.target && event.target.parentElement) || null;

    if (!target) return;

    // Find the closest matching ancestor (or the element itself)
    const matched = target.closest(selector);
    if (!matched) return;

    // Ensure the matched element is within the delegated root
    const withinRoot =
      $element === document ||
      ($element && typeof $element.contains === 'function' && $element.contains(matched)) ||
      matched === $element;

    if (!withinRoot) return;

    handler.call(matched, event);
  };

  $element.addEventListener(eventType, eventHandler, capturingPhase);

  return () => {
    $element.removeEventListener(eventType, eventHandler, capturingPhase);
  };
};

export { delegateEvent };
