import { defineConfig, loadEnv } from 'vite';
import path from 'path';
import postcssImport from 'postcss-import';
import postcssNesting from 'postcss-nesting';
import postcssCustomMedia from 'postcss-custom-media';
import autoprefixer from 'autoprefixer';
import cssnano from 'cssnano';
import fullReload from 'vite-plugin-full-reload';
import { fileURLToPath } from 'url';
import pkg from './package.json';

const __dirname = path.dirname(fileURLToPath(import.meta.url));
const THEME_NAME = pkg.name;
const SRC_PATH = path.resolve(__dirname, '.');
const DEV = process.env.NODE_ENV !== 'production';

export default defineConfig(({ mode }) => {
  const env = loadEnv(mode, process.cwd(), '');
  const proxyURL = env.PROXY_URL || 'http://localhost';

  return {
    resolve: {
      alias: {
        '~': path.resolve(__dirname, '.'),
      },
    },

    css: {
      postcss: {
        plugins: [
          postcssImport({
            resolve(id) {
              if (id.startsWith('~')) {
                return path.resolve(__dirname, id.slice(1));
              }
              return id;
            },
          }),
          postcssNesting(),
          postcssCustomMedia(),
          autoprefixer({ ignore: ['2009'] }),
          mode === 'production' && cssnano(),
        ].filter(Boolean),
      },
    },

    server: {
      origin: proxyURL,
      https: false,
      host: true,

      proxy: {
        // Proxying everything that isn't a static asset from the dev server
        '^/(?!@vite|node_modules|src|scripts|styles|components).*': {
          target: proxyURL,
          changeOrigin: true,
        },
      },
      hmr: {
        host: 'localhost',
        protocol: 'ws',
      },
    },

    build: {
      outDir: 'dist',
      emptyOutDir: true,
      cleanExclusions: ['vendor/**'],
      manifest: true,
      rollupOptions: {
        // Single entry point for JS, which will import the CSS
        input: {
          app: path.resolve(__dirname, 'scripts/app.js'),
        },
        output: {
          // Keep output paths simple and predictable
          entryFileNames: 'js/[name].js',
          assetFileNames: 'css/style.css', // We only have one CSS file
        },
      },
    },

    plugins: [fullReload(['./**/*.php', './**/*.twig'])],
  };
});
